;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/file)

;;;;
;;;;
;;;; Files and Pathnames
;;;;
;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Constants
;;;

(defining-consts
  (+mode/socket+    #o0140000)
  (+mode/symlink+   #o0120000)
  (+mode/regular+   #o0100000)
  (+mode/block+     #o0060000)
  (+mode/directory+ #o0040000)
  (+mode/char+      #o0020000)
  (+mode/fifo+      #o0010000)
  (+perm/suid+      #o0004000)
  (+perm/sgid+      #o0002000)
  (+perm/sticky+    #o0001000)
  (+perm/own-rwx+   #o0000700)
  (+perm/own-r+     #o0000400)
  (+perm/own-w+     #o0000200)
  (+perm/own-x+     #o0000100)
  (+perm/group-rwx+ #o0000070)
  (+perm/group-r+   #o0000040)
  (+perm/group-w+   #o0000020)
  (+perm/group-x+   #o0000010)
  (+perm/other-rwx+ #o0000007)
  (+perm/other-r+   #o0000004)
  (+perm/other-w+   #o0000002)
  (+perm/other-x+   #o0000001)
  (+mode/unknown+   (expt 2 64)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Generic Functions, Types, Classes
;;;

(defgeneric file-is-directory-p (file))
(defgeneric file-exists-p (file))
(defgeneric file-exists/directory-p (file))
(defgeneric file-basename (file &key &allow-other-keys))
(defgeneric file-filename (file))
(defgeneric file-size (file))
(defgeneric file-last-modified (file))
(defgeneric file-mode (file))
(defgeneric file-path (file))
(defgeneric file-symlink-p (file))

(defclass file-info ()
  ((path
    :initarg :path
    :initform ""
    :type (or string pathname))

   (size
    :initarg :size
    :initform 0
    :type integer)

   (mode
    :initarg :mode
    :initform +mode/regular+
    :type t/uint32)

   (last-modified
    :initform (get-universal-time)
    :type integer))

  (:documentation "A `FILE-INFO` is an object that holds information about a
  file.  This can be either a real file that exists somewhere on storage, or
  simply a virtual file that doesn't exist."))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Methods and Functions
;;;

(defmethod print-object ((file file-info) out)
  (print-unreadable-object (file out :type t)
    (format out " - Name: ~s, Size: ~:d bytes"
            (file-basename file)
            (file-size file))))

(defun make-file-info (pathname &optional populate)
  "Creates a new `FILE-INFO` instance using `PATHNAME` as the path to the file.
If `POPULATE` is non-`NIL`, then the resulting `FILE-INFO` instance is assumed
to actually exist, and the various fields of it will be populated before return.

This returns a new `FILE-INFO` object."
  (let ((ret (make-instance 'file-info :path pathname)))
    (when populate
      (file-size ret)
      (file-last-modified ret)
      (file-mode ret))
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod file-path ((file file-info))
  "Returns the path that is stored in `FILE`.  If `FILE` points to an existing
directory, then the returned path will always be a proper directory pathname."
  (if (file-is-directory-p file)
      (uiop:ensure-directory-pathname (slot-value file 'path))
      (slot-value file 'path)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod file-is-directory-p ((file pathname))
  "Returns the pathname if `FILE` is a pathname that actually points
to a directory, or `NIL` otherwise."
  (uiop:directory-exists-p file))

(defmethod file-is-directory-p ((file file-info))
  "Returns the path of `FILE` if it is a pathname that actually points
to a directory, or `NIL` otherwise."
  (file-is-directory-p (slot-value file 'path)))

(defmethod file-is-directory-p ((file string))
  "Returns a `PATHNAME` if `FILE` is a pathname that actually points
to a directory, or `NIL` otherwise. "
  (file-is-directory-p (uiop:parse-native-namestring file)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod file-exists/directory-p ((file pathname))
  "Check if `FILE` exists and is a directory, returning the pathname
if it does, or `NIL` otherwise."
  (uiop:directory-exists-p file))

(defmethod file-exists/directory-p ((file string))
  "Check if `FILE` exists and is a directory, returning a `PATHNAME`
equivalent to `FILE` if it does, or `NIL` otherwise."
  (file-exists/directory-p (uiop:parse-native-namestring file)))

(defmethod file-exists/directory-p ((file file-info))
  "Check if the path of `FILE` exists and is a directory, returning
the pathname if it does, or `NIL` otherwise."
  (file-exists/directory-p (slot-value file 'path)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod file-exists-p ((file pathname))
  "This checks to see if `FILE` points to an actual file that exists.
If it does, the pathname is returned.  If it does not exist, or `FILE`
actually points to a directory that exists, this returns `NIL`."
  (unless (file-exists/directory-p file)
    (probe-file file)))

(defmethod file-exists-p ((file file-info))
  "This checks to see if the path of `FILE` points to an actual file that exists.
If it does, the pathname is returned.  If it does not exist, or `FILE`
actually points to a directory that exists, this returns `NIL`."
  (file-exists-p (slot-value file 'path)))

(defmethod file-exists-p ((file string))
  "This checks to see if `FILE` points to an actual file that exists.
If it does, a `PATHNAME` equivalent to `FILE` is returned.  If it does
not exist, or `FILE` actually points to a directory that exists, this
returns `NIL`."
  (file-exists-p (uiop:parse-native-namestring file)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(declaim (inline unwilden-pathname))
(defun unwilden-pathname (pathname)
  "Returns a string where any wildcards in PATHNAME are escaped."
  (declare (type (or string pathname) pathname))
  (sdm:string-replace (sdm:string-replace (sdm:string-replace (sdm:string-replace (format nil "~a" pathname) "*" "\\*")
                                                              "?" "\\?")
                                          "[" "\\[")
                      "]" "\\]"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod file-basename ((file pathname) &key &allow-other-keys)
  "Returns the basename of `FILE`.  In other words, it returns the
final non-directory component of the path."
  (let ((ret (pathname-name file)))
    (when (pathname-type file)
      (setf ret (concatenate 'string ret "." (pathname-type file))))
    ret))

(defmethod file-basename ((file file-info) &key &allow-other-keys)
  "Returns the basename of the path of `FILE`.  In other words, it
returns the final non-directory component of the path."
  (file-basename (slot-value file 'path)))

(defmethod file-basename ((file string) &key un-wilden)
  "Returns a `PATHNAME` equivalent to the basename of `FILE`.  In
other words, it returns the final non-directory component of the
path."
  (if un-wilden
      (file-basename (native-pathname (unwilden-pathname file)))
      (file-basename (native-pathname file))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod file-filename ((file pathname))
  "Returns the filename of `FILE` without the extension as a string.  In other
words, it returns the final non-directory component of the path, without the
extension.  Wildcards are never treated as wildcards."
  (pathname-name (unwilden-pathname file)))

(defmethod file-filename ((file file-info))
  "Returns the filename of `FILE` without the extension as a string.  In other
words, it returns the final non-directory component of the path, without the
extension.  Wildcards are never treated as wildcards."
  (pathname-name (unwilden-pathname (slot-value file 'file))))

(defmethod file-filename ((file string))
  "Returns the filename of `FILE` without the extension as a string.  In other
words, it returns the final non-directory component of the path, without the
extension.  Wildcards are never treated as wildcards."
  (pathname-name (unwilden-pathname file)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun file-size/generic (file)
  (declare (type (or string pathname) file))
  (with-open-file (in file :if-does-not-exist :error)
    (file-length in)))

#+sbcl
(defun file-size/sbcl (file)
  (declare (type pathname file))
  #+unix (sb-posix:stat-size (sb-posix:stat file))
  #-unix (file-size/generic file))

(defmethod file-size ((file pathname))
  "Checks the size of `FILE` and returns two values: that size in
bytes, and `NIL`."
  (values
   #+sbcl (file-size/sbcl file)
   #-sbcl (file-size/generic file)
   nil))

(defmethod file-size ((file string))
  "Checks the size of `FILE` and returns two values: that size in
bytes, and `NIL`."
  (file-size (native-pathname file)))

(defmethod file-size ((file file-info))
  "Checks the size of the file represented by `FILE` and returns two
values.  The first value is the current size of the file in bytes.
The second value will be `NIL` if this size hasn't changed since
`FILE`'s size was last check, or the old size if it was."
  (let ((old-size (slot-value file 'size))
        (new-size (file-size (slot-value file 'path))))
    (values
     new-size
     (if (= old-size new-size)
         nil
         old-size))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod file-last-modified ((file pathname))
  "Checks when `FILE` was last modified and returns two values: that
time, and `NIL`."
  (values
   (file-write-date file)
   nil))

(defmethod file-last-modified ((file string))
  "Checks when `FILE` was last modified and returns two values: that
time, and `NIL`."
  (file-last-modified (native-pathname file)))

(defmethod file-last-modified ((file file-info))
  "Checks when the file represented by `FILE` was last modified and
returns two values.  The first value is the current modification time
of the file.  The second value will be `NIL` if this time hasn't
changed since `FILE`'s modification time was last check, or the old
time if it was."
  (let ((old-time (slot-value file 'last-modified))
        (new-time (file-last-modified (slot-value file 'path))))

    (values
     new-time
     (if (= old-time new-time)
         nil
         old-time))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun file-mode/generic (file)
  (declare (ignore file))
  +mode/unknown+)

#+sbcl
(defun file-mode/sbcl (file)
  (declare (type pathname file))
  #+unix (sb-posix:stat file)
  #-unix (file-mode/generic file))

(defmethod file-mode ((file pathname))
  "Checks the mode of `FILE` and returns two values: that mode, and
`NIL`."
  (values
   #+sbcl (file-mode/sbcl file)
   #-sbcl (file-mode/generic file)
   nil))

(defmethod file-mode ((file string))
  "Checks the mode of `FILE` and returns two values: that mode, and
`NIL`."
  (file-mode (native-pathname file)))

(defmethod file-mode ((file file-info))
  "Checks the mode of the file represented by `FILE` and returns two
values.  The first value is the current mode of the file.  The second
value will be `NIL` if this mode hasn't changed since `FILE`'s mode
was last check, or the old mode if it was."
  (let ((old-mode (slot-value file 'mode))
        (new-mode (file-mode (slot-value file 'path))))

    (values
     new-mode
     (if (= old-mode new-mode)
         nil
         old-mode))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Other Functions
;;;

(defun get-temporary-filename (prefix suffix &optional (max-tries 42))
  "Attempts to construct a temporary filename that doesn't yet exist.
This filename will be prefixed with PREFIX, suffixed with SUFFIX, and
the middle portion will be a hashed value dependent on the current
date and time, the internal run time, and possibly an extra letter.

The filename that's generated will always exist within the system's
standard temporary directory.

This returns the new filename on success, or will raise an `ERROR` if
it could not succeed."
  (let ((extra-letters (loop for i from (char-code #\A) to (char-code #\Z)
                          collect (code-char i))))

    (dolist (extra-letter extra-letters)
      (dotimes (i max-tries)
        (multiple-value-bind (second minute hour day month year)
            (decode-universal-time (get-universal-time))

          (let ((datestr (format nil "DATETIME-~a-~a-~a ~a:~a:~a^_^~a~a~a"
                                 year month day hour minute second
                                 (get-internal-real-time)
                                 #+sbcl (sb-sys:get-system-info)
                                 #-sbcl (+ (random 42) (get-internal-run-time))
                                 extra-letter)))

            (loop for c across datestr
               summing (char-code c) into ret
               finally
                 (let ((ret (uiop:merge-pathnames*
                             uiop:*temporary-directory*
                             (format nil "~a~a~a" prefix (sxhash ret) suffix))))
                   (with-open-file (out ret :direction :output
                                        :if-does-not-exist :create
                                            :if-exists :error)
                     (declare (ignorable out)))
                   (return-from get-temporary-filename ret))))))))

  (error "Could not generate a temporary filename"))

(defmethod file-symlink-p ((file pathname))
  "Check if `FILE` exists and is a directory, returning the pathname
if it does, or `NIL` otherwise."
  #+sbcl (sb-posix:s-islnk (sb-posix:stat-mode (sb-posix:lstat file)))
  #-sbcl (error "Cannot yet determine if something is a symlink on this Lisp implementation"))

(defmethod file-symlink-p ((file string))
  "Check if `FILE` exists and is a directory, returning a `PATHNAME`
equivalent to `FILE` if it does, or `NIL` otherwise."
  (file-symlink-p (native-pathname file)))

(defmethod file-symlink-p ((file file-info))
  "Check if the path of `FILE` exists and is a directory, returning
the pathname if it does, or `NIL` otherwise."
  (file-symlink-p (slot-value file 'path)))

(define-typed-fn native-namestring (thing)
    (string t)
  "Returns a native namestring for THING.  If THING is a STRING, then it is first
rendered to a PATHNAME using UIOP:PARSE-NATIVE-NAMESTRING, and then rendered
back to a namestring.

THING must be a PATHNAME or a STRING."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (nth-value
   0
   (etypecase thing
     (string (uiop:native-namestring (uiop:parse-native-namestring thing)))
     (pathname (uiop:native-namestring thing)))))

(define-typed-fn native-pathname (thing)
    (pathname t)
  "Returns a PATHNAME for THING, handling native namestrings as-needed.

THING must be a PATHNAME or a STRING."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (nth-value
   0
   (etypecase thing
     (string (uiop:parse-native-namestring thing))
     (pathname (uiop:parse-native-namestring (uiop:native-namestring thing))))))
