;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;
;;;; Fast IEEE 754 Float Conversions (ClozureCL)
;;;;
;;;; Uses FFI to do C Union-like conversions.
;;;;

(ccl:def-foreign-type alien-float->uint
    (:union alien-float->uint
            (int-val (:unsigned 32))
            (float-val (:single-float))))

(ccl:def-foreign-type alien-double->uint
    (:union alien-double->uint
            (int-val (:unsigned 64))
            (float-val :double-float)))

(defmacro %ccl-with-alien ((name type) &body forms)
  `(let ((,name (ccl:make-record ,type)))
     (unwind-protect
          (progn ,@forms)
       (ccl:free ,name))))

(define-typed-fn fast-sfloat->uint32 ((single-float value))
    (t/uint32 t)
  "Encodes a SINGLE-FLOAT as a T/UINT32.  This is the 'fast' version that uses a
union type internally."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (%ccl-with-alien (alien alien-float->uint)
    (setf (ccl:pref alien :alien-float->uint.float-val) value)
    (ccl:pref alien :alien-float->uint.int-val)))

(define-typed-fn fast-uint32->sfloat ((t/uint32 value))
    (single-float t)
  "Decodes a T/UINT32 to a SINGLE-FLOAT.  This is the 'fast' version that uses a
union type internally."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (%ccl-with-alien (alien alien-float->uint)
    (setf (ccl:pref alien :alien-float->uint.int-val) value)
    (ccl:pref alien :alien-float->uint.float-val)))

(define-typed-fn fast-dfloat->uint64 ((double-float value))
    (t/uint64 t)
  "Encodes a DOUBLE-FLOAT as a T/UINT64.  This is the 'fast' version that uses a
union type internally."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (%ccl-with-alien (alien alien-double->uint)
    (setf (ccl:pref alien :alien-double->uint.float-val) value)
    (ccl:pref alien :alien-double->uint.int-val)))

(define-typed-fn fast-uint64->dfloat ((t/uint64 value))
    (double-float t)
  "Decodes a T/UINT64 to a DOUBLE-FLOAT.  This is the 'fast' version that uses a
union type internally."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (%ccl-with-alien (alien alien-double->uint)
    (setf (ccl:pref alien :alien-double->uint.int-val) value)
    (ccl:pref alien :alien-double->uint.float-val)))
