;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-crc)

;;;;
;;;; 64-bit CRC
;;;;

(declaim (type t/uint64 +crc64-ecma-polynomial+ +crc64-iso-polynomial+))
(defining-consts
  (+crc64-ecma-polynomial+ #xC96C5795D7870F42)
  (+crc64-iso-polynomial+ #xD800000000000000))

(defstruct (crc64 (:constructor %make-crc64)
                  (:conc-name %crc64-))
  "A CRC64 is a record that holds state for computing a CRC64 from one or more values."
  (crc 0 :type t/uint64)
  (poly 0 :type t/uint64)
  (table (new-array 256 t/uint64) :type (simple-array t/uint64 (256))))

(defmethod print-object ((obj crc64) out)
  (print-unreadable-object (obj out :type t)
    (format out "CRC: ~16,'0x, Polynomial: ~16,'0x" (%crc64-crc obj) (%crc64-poly obj))))

(defun make-crc64 (&optional (polynomial +crc64-ecma-polynomial+))
  "Creates a new CRC64 instance."
  (check-type polynomial t/uint64)
  (let ((ret (%make-crc64)))
    (setf (crc64-polynomial ret) polynomial)
    ret))

(define-typed-fn crc64-crc ((crc64 crc))
    (t/uint64 t)
  "Returns the current computed CRC."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (%crc64-crc crc))

(define-typed-fn crc64-reset ((crc64 crc))
    (crc64 t)
  "Resets the current CRC to 0.  This is the same as starting anew."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (setf (%crc64-crc crc) 0)
  crc)

(define-typed-fn crc64-polynomial ((crc64 crc))
    (t/uint64 t)
  "Returns the current polynomial that is being used for CRC64 calculations."
  (declare (optimize (speed 3) (debug 1) (safety 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (%crc64-poly crc))

(defun (setf crc64-polynomial) (poly crc)
  "Sets the polynomial that will be used for CRC64 calculations.  This also
resets the current CRC value to 0."
  (declare (type t/uint64 poly)
           (type crc64 crc)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (crc64-reset crc)
  (setf (%crc64-poly crc) poly)
  (loop for i fixnum from 0 below 256 do
    (let ((val i))
      (declare (type t/uint64 val))
      (dotimes (j 8)
        (if (/= (logand val 1) 0)
            (setf val (logxor (ash val -1) poly))
            (setf val (ash val -1))))
      (setf (aref (%crc64-table crc) i) val)))
  crc)

(defgeneric crc64-update (crc value)
  (:documentation "Updates the CRC using VALUE.  The VALUE parameter can be an
8-bit unsigned integer, a string, a vector of 8-bit unsigned integers, or a list
of 8-bit unsigned integers.  Returns the new computed CRC value."))

(defmethod crc64-update ((crc crc64) (value integer))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (check-type value t/uint8)
  (let ((acc (%crc64-crc crc)))
    (setf acc (logand #xFFFFFFFFFFFFFFFF (lognot acc)))
    (setf acc (logxor (aref (%crc64-table crc) (logxor value (logand acc #xFF))) (ash acc -8)))
    (setf (%crc64-crc crc) (logand #xFFFFFFFFFFFFFFFF (lognot acc)))))

(defmethod crc64-update ((crc crc64) (value string))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (loop for c character across value
        for ret = (crc64-update crc (char-code c))
        finally (return ret)))

(defmethod crc64-update ((crc crc64) (value vector))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (loop for x fixnum across value
        for ret = (crc64-update crc x)
        finally (return ret)))

(defmethod crc64-update ((crc crc64) (value list))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (loop for x fixnum in value
        for ret = (crc64-update crc x)
        finally (return ret)))

(defmacro with-crc64 ((crc &optional (polynomial +crc64-ecma-polynomial+)) &body forms)
  "Creates a new CRC64 instance with the given polynomial and binds it to CRC.
This then executes FORMS.  At the end, this returns the final CRC value."
  `(let ((,crc (make-crc64 ,polynomial)))
     ,@forms
     (crc64-crc ,crc)))
