;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-config)

(defun t/directory-type-p (thing)
  (or (and (keywordp thing)
           (or (eq thing :xdg)
               (eq thing :windows)))
      (and (listp thing)
           (= (length thing) 5)
           (eq (car thing) :custom)
           (or (pathnamep (getf (cdr thing) :config-dir))
               (stringp (getf (cdr thing) :config-dir)))
           (or (pathnamep (getf (cdr thing) :data-dir))
               (stringp (getf (cdr thing) :data-dir))))))

(deftype t/directory-type ()
  '(satisfies t/directory-type-p))

(defgeneric define-data-file (res symbolic-name filename)
  (:documentation "Associates a filename that will appear in the data directory
with a symbolic name in the given RESOLVER.  The symbolic name can be a symbol,
keyword, or string.

When calling this on a PRIORITY-RESOLVER, then the data file is defined for each
RESOLVER contained within it."))

(defgeneric resolver-get-data-file (res symbolic-name)
  (:documentation "Retrieves the filename associated with SYMBOLIC-NAME from the
given RESOLVER's data directory.  If the data file is not defined, this returns
NIL.  This accepts both RESOLVERs and PRIORITY-RESOLVERs."))

(defgeneric resolver-get-data-file! (res symbolic-name)
  (:documentation "Retrieves the filename associated with SYMBOLIC-NAME from the
given RESOLVER's data directory.  If the data file is not defined, this raises
an UNKNOWN-DATA-FILE condition.  This condition is restartable.  This generic
function accepts both RESOLVERs and PRIORITY-RESOLVERs."))

(defgeneric resolver-get-data-file-stream (res symbolic-name &key element-type direction
                                                               if-exists if-does-not-exist external-format)
  (:documentation "Looks up the data file associated with SYMBOLIC-NAME, then
opens it and returns the STREAM.  The keyword arguments are all the same as for
OPEN.  If the data file is not defined, this returns NIL.  This generic function
accepts both RESOLVERs and PRIORITY-RESOLVERs."))

(defgeneric resolver-get-data-file-stream! (res symbolic-name &key element-type direction
                                                                if-exists if-does-not-exist external-format)
  (:documentation "Looks up the data file associated with SYMBOLIC-NAME, then
opens it and returns the STREAM.  The keyword arguments are all the same as for
OPEN.  If the data file is not defined, this raises an UNKNOWN-DATA-FILE
condition.  This condition is restartable.  This generic function accepts both
RESOLVERs and PRIORITY-RESOLVERs."))

(define-condition unknown-data-file (simple-error)
  ((symbolic-name
    :initarg :symbolic-name
    :initform nil
    :reader error-symbolic-name)))

(define-condition file-does-not-exist (file-error)
  ()
  (:report (lambda (err out)
             (format out "File does not exist: ~a" (file-error-pathname err)))))

(define-condition missing-data-file (file-does-not-exist)
  ((symbolic-name
    :initarg :symbolic-name
    :initform nil
    :reader error-symbolic-name)))

(defclass resolver ()
  ((type
    :initarg :type
    :initform :xdg
    :type t/directory-type
    :reader resolver-type)

   (base-name
    :initarg :base-name
    :initform ""
    :type string
    :reader resolver-base-name
    :documentation "The base name for this configuration and data set.  This is
typically the name of the program.  This slot must be set on initialization, and
cannot be an empty string.")

   (data-dir
    :initform nil
    :reader resolver-data-dir
    :documentation "The directory where data files are stored for programs.
This will be combined with the BASE-NAME to create a full data path when
resolving a data filename.

This is automatically set when a RESOLVER instance is created based on the TYPE.")

   (config-dir
    :initform nil
    :reader resolver-config-dir
    :documentation "The directory where configuration files are stored for
programs.  This will be combined with the BASE-NAME to create a full config path
when resolving a config filename.

This is automatically set when a RESOLVER instance is created based on the TYPE.")

   (%data-files
    :initform (make-hash-table :test 'equal)
    :type hash-table)))

(defun resolver-p (thing)
  (declare (optimize (speed 3) (debug 1) (safety 1)))
  (typep thing 'resolver))

(defun resolver-list-p (thing)
  (declare (optimize (speed 3) (debug 1) (safety 1)))
  (and (listp thing)
       (every #'resolver-p thing)))

(defmethod initialize-instance :after ((res resolver) &key &allow-other-keys)
  (with-slots (base-name data-dir config-dir type) res
    (check-type type t/directory-type)
    (check-type base-name string)

    (when (empty-string-p base-name)
      (error "RESOLVER base name cannot be empty"))

    (if (listp type)
        (if (eq (car type) :custom)
            (progn
              (unless (setf config-dir (getf (cdr type) :config-dir))
                (error ":CUSTIOM type specified, but no :CONFIG-DIR specified"))
              (unless (setf data-dir (getf (cdr type) :data-dir))
                (error ":CUSTIOM type specified, but no :DATA-DIR specified"))

              (setf config-dir (uiop:ensure-directory-pathname config-dir))
              (setf data-dir (uiop:ensure-directory-pathname data-dir)))
            (error ":TYPE specified as a list, but the first value was not :TYPE"))

        (ecase type
          (:xdg
           (setf config-dir (uiop:ensure-directory-pathname (uiop:xdg-config-home base-name)))
           (setf data-dir (uiop:ensure-directory-pathname (uiop:xdg-data-home base-name))))

          (:windows
           (setf config-dir (uiop:ensure-directory-pathname
                             (uiop:merge-pathnames* base-name (uiop:get-folder-path :appdata))))
           (setf config-dir (uiop:ensure-directory-pathname
                             (uiop:merge-pathnames* base-name (uiop:get-folder-path :appdata)))))

          (:custom
           (error ":CUSTOM cannot be used on its own, and must be used as a list, e.g. ~
(:CUSTOM :CONFIG-DIR \"/path1\" :DATA-DIR \"/path2\")"))))))

(defun make-resolver (base-name &key (type :xdg) dont-create-missing-directories)
  "Creates a new RESOLVER instance."
  (let ((ret (make-instance 'resolver :base-name base-name :type type)))
    (with-slots (data-dir config-dir) ret
      (unless dont-create-missing-directories
        (ensure-directories-exist config-dir)
        (ensure-directories-exist data-dir)))
    ret))

(defmethod define-data-file ((res resolver) symbolic-name (filename string))
  (define-data-file res symbolic-name (sdm-file:native-pathname filename)))

(defmethod define-data-file ((res resolver) (symbolic-name symbol) (filename pathname))
  (with-typed-slots ((hash-table %data-files)) res
    (setf (gethash symbolic-name %data-files) filename))
  symbolic-name)

(defmethod define-data-file ((res resolver) (symbolic-name string) (filename pathname))
  (with-typed-slots ((hash-table %data-files)) res
    (setf (gethash symbolic-name %data-files) filename))
  symbolic-name)

(defun %resolver-get-data-file-pathname (res symname)
  (declare (type resolver res))
  (with-typed-slots ((hash-table %data-files)) res
    (multiple-value-bind (ret found)
        (gethash symname %data-files)
      (when found
        (uiop:merge-pathnames* ret (resolver-data-dir res))))))

(defun %resolver-get-data-file-pathname! (res symname)
  (declare (type resolver res))
  (or (%resolver-get-data-file-pathname res symname)
      (error 'unknown-data-file :symbolic-name symname
                                :format-control "Data file is not known: ~a"
                                :format-arguments (list symname))))

(defmethod resolver-get-data-file ((res resolver) (symbolic-name symbol))
  (%resolver-get-data-file-pathname res symbolic-name))

(defmethod resolver-get-data-file ((res resolver) (symbolic-name string))
  (%resolver-get-data-file-pathname res symbolic-name))

(defun %read-new-filename ()
  (format t "Enter the filename to use: ")
  (multiple-value-list (eval (read))))

(defmethod resolver-get-data-file! ((res resolver) (symbolic-name symbol))
  (restart-case
      (%resolver-get-data-file-pathname! res symbolic-name)
    (define-the-file (filename)
      :report "Define the data file"
      :interactive %read-new-filename
      (define-data-file res symbolic-name filename)
      (return-from resolver-get-data-file! (resolver-get-data-file res symbolic-name)))))

(defmethod resolver-get-data-file! ((res resolver) (symbolic-name string))
  (restart-case
      (%resolver-get-data-file-pathname! res symbolic-name)
    (define-the-file (filename)
      :report "Define the data file"
      :interactive %read-new-filename
      (define-data-file res symbolic-name filename)
      (return-from resolver-get-data-file! (resolver-get-data-file res symbolic-name)))))

(defmacro with-open-data-file ((var res symbolic-name
                                &key (element-type 'base-char) (direction :input)
                                  if-exists if-does-not-exist (external-format :default))
                               &body forms)
  "Looks up the data file associated with SYMBOLIC-NAME, then opens it and binds
the stream to VAR.  FORMS is then executed.  The stream is closed when FORMS
finishes.

If the data file is not defined, this raises an UNKNOWN-DATA-FILE condition.
This condition is restartable.

The keyword arguments are all the same as for OPEN."
  `(with-open-file (,var (resolver-get-data-file! ,res ,symbolic-name)
                         :direction ,direction :element-type (quote ,element-type)
                         :if-exists ,if-exists :if-does-not-exist ,if-does-not-exist
                         :external-format ,external-format)
     ,@forms))

(defun %resolver-get-data-file-stream (res symname retrieve-fn element-type direction if-exists if-does-not-exist
                                       external-format)
  (declare (type resolver res)
           (type function retrieve-fn))
  (let ((filename (funcall retrieve-fn res symname)))
    (open filename :direction direction :element-type element-type
                   :if-exists if-exists :if-does-not-exist if-does-not-exist
                   :external-format external-format)))

(defmethod resolver-get-data-file-stream ((res resolver) symbolic-name
                                          &key (element-type 'base-char) (direction :input)
                                            if-exists (if-does-not-exist :error) (external-format :default))
  (%resolver-get-data-file-stream res symbolic-name #'resolver-get-data-file element-type direction if-exists
                                  if-does-not-exist external-format))

(defmethod resolver-get-data-file-stream! ((res resolver) symbolic-name
                                           &key (element-type 'base-char) (direction :input)
                                             if-exists if-does-not-exist (external-format :default))
  (restart-case
      (%resolver-get-data-file-stream res symbolic-name #'resolver-get-data-file! element-type direction if-exists
                                      if-does-not-exist external-format)
    (define-the-file (filename)
      :report "Define the data file"
      :interactive %read-new-filename
      (define-data-file res symbolic-name filename)
      (return-from resolver-get-data-file-stream!
        (%resolver-get-data-file-stream res symbolic-name #'resolver-get-data-file! element-type direction if-exists
                                        if-does-not-exist external-format)))))
