;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-config)

(defclass priority-resolver ()
  ((resolvers
    :initarg :resolvers
    :initform nil
    :type list
    :reader priority-resolver-resolvers))
  (:documentation "The PRIORITY-RESOLVER class is used to store a set of
RESOLVERs.  When running the RESOLVER protocol on a PRIORITY-RESOLVER instance,
the RESOVLERs in this instance are checked in the order they're defined for the
requested file.  This allows the bundling of multiple resolvers.

A usage example might having one RESOLVER each for: the current directory, the
user's XDG directories, and the system-wide directories.  When defined in this
order, a call to (for example) READ-CONFIG-INSTANCE would first check the
current directory for the config file, then check the XDG directory if it
doesn't exist, then finally the system-wide directory, all in a single call."))

(defgeneric priority-resolver-add (priority-res res)
  (:documentation "Adds a new RESOLVER to the given PRIORITY-RESOLVER.  The new
RESOLVER is always set to the lowest priority.  This returns the number of
RESOLVERs in the PRIORITY-RESOLVER."))

(defgeneric priority-resolver-insert (priority-res res at)
  (:documentation "Adds a new RESOLVER to the given PRIORITY-RESOLVER at the
given priority.  If the priority is less than or equal to zero, then it is
always inserted as the top priority.  If the priority is greater than the number
of RESOLVERs in the PRIORITY-RESOLVER, then it is always added at the lowest
priority.  This returns the number of RESOLVERs in the PRIORITY-RESOLVER."))

(defgeneric priority-resolver-remove (priority-res res)
  (:documentation "Removes the given RESOLVER from a PRIORITY-RESOLVER.  This
returns the number of RESOLVERs in the PRIORITY-RESOLVER.  RES can be either a
RESOLVER instance, or an index."))

(defun make-priority-resolver (&rest resolvers)
  "Creates a new PRIORITY-RESOLVER instance with the given RESOLVERs.  The order
of the RESOVLERs is important, with the first RESOLVER taking highest priority."
  (assert (resolver-list-p resolvers))
  (make-instance 'priority-resolver :resolvers resolvers))

(defmethod define-data-file ((res priority-resolver) symbolic-name filename)
  (dolist (res (the list (priority-resolver-resolvers res)))
    (define-data-file res symbolic-name filename)))

(defun %priority-resolver-get-data-file-pathname (priority-res symname)
  (declare (type priority-resolver priority-res))
  (loop for res in (the list (priority-resolver-resolvers priority-res))
        for file = (%resolver-get-data-file-pathname res symname)
        when file do (return file)))

(defun %priority-resolver-get-data-file-pathname! (priority-res symname)
  (declare (type priority-resolver priority-res))
  (or (%priority-resolver-get-data-file-pathname priority-res symname)
      (error 'unknown-data-file :symbolic-name symname
                                :format-control "Data file is not known in any resolvers: ~a"
                                :format-arguments (list symname))))

(defun %priority-resolver-get-data-file-stream (res symname retrieve-fn element-type direction
                                                if-exists if-does-not-exist external-format)
  (declare (type priority-resolver res)
           (type function retrieve-fn))
  (let ((filename (funcall retrieve-fn res symname)))
    (open filename :direction direction :element-type element-type
                   :if-exists if-exists :if-does-not-exist if-does-not-exist
                   :external-format external-format)))

(defmethod resolver-get-data-file ((res priority-resolver) symbolic-name)
  (%priority-resolver-get-data-file-pathname res symbolic-name))

(defmethod resolver-get-data-file! ((res priority-resolver) symbolic-name)
  (restart-case
      (%priority-resolver-get-data-file-pathname! res symbolic-name)
    (define-the-file (filename)
      :report "Define the data file for all resolvers"
      :interactive %read-new-filename
      (define-data-file res symbolic-name filename)
      (return-from resolver-get-data-file! (resolver-get-data-file res symbolic-name)))))

(defmethod resolver-get-data-file-stream ((res priority-resolver) symbolic-name
                                          &key (element-type 'base-char) (direction :input)
                                            if-exists (if-does-not-exist :error) (external-format :default))
  (%priority-resolver-get-data-file-stream res symbolic-name #'resolver-get-data-file element-type direction
                                           if-exists if-does-not-exist external-format))

(defmethod resolver-get-data-file-stream! ((res priority-resolver) symbolic-name
                                           &key (element-type 'base-char) (direction :input)
                                             if-exists if-does-not-exist (external-format :default))
  (restart-case
      (%priority-resolver-get-data-file-stream res symbolic-name #'resolver-get-data-file! element-type direction
                                               if-exists if-does-not-exist external-format)
    (define-the-file (filename)
      :report "Define the data file for all resolvers"
      :interactive %read-new-filename
      (define-data-file res symbolic-name filename)
      (return-from resolver-get-data-file-stream!
        (%resolver-get-data-file-stream res symbolic-name #'resolver-get-data-file! element-type direction if-exists
                                        if-does-not-exist external-format)))))

(defmethod priority-resolver-add ((priority-res priority-resolver) (res resolver))
  (length (setf (slot-value priority-res 'resolvers)
                (append (slot-value priority-res 'resolvers)
                        (list res)))))

(defmethod priority-resolver-insert ((priority-res priority-resolver) (res resolver) (at integer))
  (with-typed-slots ((list resolvers))
      priority-res
    (cond
      ((<= at 0)
       (setf resolvers (append (list res) resolvers)))

      ((>= at (length resolvers))
       (setf resolvers (append resolvers (list res))))

      (t
       (setf resolvers (append (subseq resolvers 0 at) (list res) (subseq resolvers at)))))
    (length resolvers)))

(defmethod priority-resolver-remove ((priority-res priority-resolver) (res integer))
  (with-typed-slots ((list resolvers))
      priority-res
    (cond
      ((or (< res 0)
           (>= res (length resolvers)))
       (error "Index out of range for ~:d resolvers: ~:d" (length resolvers) res))

      ((= res (1- (length resolvers)))
       ;; Remove last one
       (setf resolvers (nbutlast resolvers)))

      ((= res 0)
       ;; Remove the first
       (pop resolvers))

      (t
       (setf resolvers (append (subseq resolvers 0 res) (subseq resolvers (1+ res))))))
    (length resolvers)))

(defmethod priority-resolver-remove ((priority-res priority-resolver) (res resolver))
  (let ((pos (position res (priority-resolver-resolvers priority-res) :test #'eq)))
    (if pos
        (priority-resolver-remove priority-res pos)
        (error "RESOLVER ~a is not part of that PRIORITY-RESOLVER." res))))
