;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-config)

(defparameter *known-support-formats* (make-hash-table))

(defstruct support-format-def
  (class    nil :type (or null symbol))
  (read-fn  nil :type (or null function))
  (write-fn nil :type (or null function)))

(defun %register-support-format (name class read-instance-fn write-instance-fn)
  (check-type name keyword)
  (check-type class symbol)
  (check-type read-instance-fn function)
  (check-type write-instance-fn function)

  (multiple-value-bind (old-class found)
      (gethash name *known-support-formats*)
    (if found
        (restart-case
            (error "Support format ~a already registered and associated with ~a" name old-class)
          (overwrite ()
            :report "Overwrite the old definition"
            (setf (gethash name *known-support-formats*)
                  (make-support-format-def
                   :class class
                   :read-fn read-instance-fn
                   :write-fn write-instance-fn))))
        (setf (gethash name *known-support-formats*)
              (make-support-format-def
               :class class
               :read-fn read-instance-fn
               :write-fn write-instance-fn)))))

(defun get-known-support-formats ()
  "Returns all of the known support formats as a LIST of CONSes."
  (loop for name being the hash-key in *known-support-formats*
          using (hash-value data)
        collect (cons name (support-format-def-class data))))

(defun get-support-format-class (name)
  "Looks up NAME to see if it's a known support format.  If it is, this returns
the metaclass associated with that format.  Otherwise this returns NIL."
  (multiple-value-bind (ret found)
      (gethash name *known-support-formats*)
    (when found
      (support-format-def-class ret))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defgeneric get-config-instance (res type &rest init-args)
  (:documentation "Gets a configuration file of TYPE, then returns a new
instance of that type.  TYPE must be a class that uses one of the known
support formats as a metaclass.

INIT-ARGS will be passed to MAKE-INSTANCE and used as initialization
arguments.

This accepts both RESOLVER and PRIORITY-RESOLVER instances."))

(defgeneric read-config-instance (res type)
  (:documentation "Gets a configuration file of TYPE, then returns a new
instance of that type by reading the associated configuration file.  If the file
does not exist, this returns NIL.  TYPE must be a class that uses one of the
known support formats as a metaclass.  This accepts both RESOLVER and
PRIORITY-RESOLVER instances."))

(defgeneric read-config-instance! (res type)
  (:documentation "Gets a configuration file of TYPE, then returns a new
instance of that type by reading the associated configuration file.  If the file
does not exist, a FILE-DOES-NOT-EXIST condition is raised.  TYPE must be a class
that uses one of the known support formats as a metaclass.

This does not accept PRIORITY-RESOLVER instances and must be used with a normal
RESOLVER."))

(defgeneric write-config-instance (res object &key if-exists)
  (:documentation "Writes the configuration object OBJECT to the appropriate
configuration file.  OBJECT must use a metaclass that is a registered support
class.  IF-EXISTS is the same as for OPEN.

This does not accept PRIORITY-RESOLVER instances and must be used with a normal
RESOLVER."))

(defgeneric read-or-make-config-instance (res type &rest init-args)
  (:documentation "Attempts to load a configuration file of TYPE.  If the
configuration file doesn't exist, this instead returns a new instance of TYPE.
This returns two values, the new instance of TYPE, and either T if the instance
was loaded from a file or NIL if it was created fresh using INIT-ARGS.

TYPE must be a class that uses one of the known support formats as a metaclass.
INIT-ARGS will be passed to MAKE-INSTANCE and used as initialization arguments.

This accepts both RESOLVER and PRIORITY-RESOLVER instances."))

(defgeneric config-filename (res type)
  (:documentation "Returns the full path for a config file of the given type."))

(defclass config-file (standard-class)
  ((filename
    :initarg :filename
    :initform nil
    :reader config-file-filename))
  (:documentation "A metaclass that defines a support format that can be used as
a configuration file.  This should never be used directly, but instead one of
its subclasses."))

(defun %supported-metaclass-p (metaclass)
  "Checks to see if METACLASS is a known subclass of CONFIG-FILE."
  (maphash #'(lambda (k v)
               (declare (ignore k))
               (when (closer-mop:subclassp metaclass (support-format-def-class v))
                 (return-from %supported-metaclass-p t)))
           *known-support-formats*)
  nil)

(defun %read-fn-for-metaclass (metaclass)
  "Checks to see if METACLASS is a known subclass of CONFIG-FILE."
  (maphash #'(lambda (k v)
               (declare (ignore k))
               (when (closer-mop:subclassp metaclass (support-format-def-class v))
                 (return-from %read-fn-for-metaclass (support-format-def-read-fn v))))
           *known-support-formats*)
  nil)

(defun %write-fn-for-metaclass (metaclass)
  "Checks to see if METACLASS is a known subclass of CONFIG-FILE."
  (maphash #'(lambda (k v)
               (declare (ignore k))
               (when (closer-mop:subclassp metaclass (support-format-def-class v))
                 (return-from %write-fn-for-metaclass (support-format-def-write-fn v))))
           *known-support-formats*)
  nil)

(defmethod get-config-instance ((res resolver) (type symbol) &rest init-args)
  (if (%supported-metaclass-p (class-of (find-class type)))
      (apply #'make-instance type init-args)
      (error "Class ~a does not use a known support format metaclass" type)))

(defmethod get-config-instance ((res resolver) (type class) &rest init-args)
  (get-config-instance res (class-name type) init-args))

(defmethod get-config-instance ((res priority-resolver) type &rest init-args)
  ;; The config format is the same for all resolvers, so it doesn't matter which
  ;; one we use.
  (get-config-instance (car (priority-resolver-resolvers res)) type init-args))

(defmethod read-config-instance ((res resolver) (type symbol))
  (if (%supported-metaclass-p (class-of (find-class type)))
      (let ((read-fn (%read-fn-for-metaclass (class-of (find-class type))))
            (filename (config-file-filename (find-class type))))
        (unless read-fn
          (error "Configuration support class ~a does not have a read function" type))

        (setf filename (uiop:merge-pathnames* (if (listp filename) (car filename) filename)
                                              (resolver-config-dir res)))
        (when (probe-file filename)
          (funcall read-fn filename type)))
      (error "Class ~a does not use a known support format metaclass" type)))

(defmethod read-config-instance ((res priority-resolver) type)
  (loop for sub-res in (the list (priority-resolver-resolvers res))
        for ret = (read-config-instance sub-res type)
        when ret do (return ret)))

(defmethod read-config-instance! ((res resolver) (type symbol))
  (if (%supported-metaclass-p (class-of (find-class type)))
      (let ((read-fn (%read-fn-for-metaclass (class-of (find-class type))))
            (filename (config-file-filename (find-class type))))
        (unless read-fn
          (error "Configuration support class ~a does not have a read function" type))

        (setf filename (uiop:merge-pathnames* (if (listp filename) (car filename) filename)
                                              (resolver-config-dir res)))
        (if (probe-file filename)
            (read-config-instance res type)
            (error 'file-does-not-exist :pathname filename)))

      (error "Class ~a does not use a known support format metaclass" type)))

(defmethod read-config-instance ((res resolver) (type class))
  (read-config-instance res (class-name type)))

(defmethod write-config-instance ((res resolver) object &key (if-exists :error))
  (let ((type (class-of object)))
    (if (%supported-metaclass-p (class-of type))
        (let ((write-fn (%write-fn-for-metaclass (class-of type)))
              (filename (config-file-filename type)))
          (unless write-fn
            (error "Configuration support class ~a does not have a write function" type))

          (setf filename (uiop:merge-pathnames* (if (listp filename) (car filename) filename)
                                                (resolver-config-dir res)))
          (funcall write-fn filename object :if-exists if-exists))
        (error "Class ~a does not use a known support format metaclass" type))))

(defun %read-or-make-config-instance (res type init-args)
  (let ((ret (read-config-instance res type)))
    (if ret
        (values ret t)
        (values (apply #'get-config-instance res type init-args) nil))))

(defmethod read-or-make-config-instance ((res resolver) type &rest init-args)
  (%read-or-make-config-instance res type init-args))

(defmethod read-or-make-config-instance ((res priority-resolver) type &rest init-args)
  (%read-or-make-config-instance res type init-args))

(defmethod config-filename ((res resolver) (type class))
  (sdm-file:native-pathname (uiop:merge-pathnames*
                             (sdm-file:native-pathname (config-file-filename type))
                             (resolver-config-dir res))))
