;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; Copyright (C) 2022 Jaime Olivares
;;;; Copyright (C) 2015 Jaime Olivares
;;;; Copyright (c) 2011 Matthew Francis
;;;; Ported from the Java implementation by Matthew Francis:
;;;; https://github.com/MateuszBartosiewicz/bzip2.
;;;;
;;;; Ported by Remilia Scarlet from the C# implementation by Jamie Olivares:
;;;; http://github.com/jaime-olivares/bzip2
;;;;
;;;; Modified by drone1400
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-bzip2)

(defgeneric %writer-init-next-block (writer))
(defgeneric %writer-compress-block (writer))
(defgeneric %writer-finish (writer))

(defclass writer (trivial-gray-streams:fundamental-binary-output-stream)
  ((output
    :type stream)

   (bit-writer
    :type bit-writer)

   (stream-block-size
    :type t/int32)

   (stream-crc
    :initform 0
    :type t/uint32)

   (int32-pool
    :initform (make-array-pool 't/int32 0)
    :type array-pool)

   (block-comp
    :initform nil
    :type (or null block-compressor))

   (closed?
    :initform nil
    :type boolean)

   (sync-close?
    :type boolean
    :accessor writer-sync-close-p)))

(defun make-writer (output-stream &key (level +default-compression-level+) sync-close)
  (check-type output-stream stream)
  (check-type level fixnum)
  (let ((ret (make-instance 'writer))
        (real-level (clamp level +min-compression-level+ +max-compression-level+)))
    (setf (slot-value ret 'output) output-stream)
    (setf (slot-value ret 'stream-block-size)
          (* real-level 100000))
    (setf (slot-value ret 'bit-writer) (make-instance 'bit-writer :io output-stream))
    (setf (slot-value ret 'sync-close?) sync-close)

    (with-slots (bit-writer)
        ret
      (bit-writer-write-bits bit-writer 16 +stream-start-marker-1+)
      (bit-writer-write-bits bit-writer 8  +stream-start-marker-2+)
      (bit-writer-write-bits bit-writer 8  (coerce-to-uint32 (+ (char-code #\0) real-level))))

    (%writer-init-next-block ret)
    ret))

(defmethod stream-element-type ((stream writer))
  (declare (optimize speed (debug 0) (safety 0)))
  '(unsigned-byte 8))

(defmethod trivial-gray-streams:stream-finish-output ((stream writer))
  (declare (optimize speed (debug 0) (safety 0)))
  (finish-output (slot-value stream 'output)))

(defmethod trivial-gray-streams:stream-force-output ((stream writer))
  (declare (optimize speed (debug 0) (safety 0)))
  (force-output (slot-value stream 'output)))

(defmethod close ((stream writer) &key abort)
  (declare (ignore abort)
           (optimize speed (debug 1)))
  (with-typed-slots ((boolean closed? sync-close?)
                     (stream output))
      stream
    (unless (or closed? (not (open-stream-p output)))
      (%writer-finish stream) ;; Sets CLOSED? to T
      (when sync-close?
        (close output))
      t)))

(defmethod open-stream-p ((stream writer))
  (declare (optimize speed (debug 0) (safety 0)))
  (not (slot-value stream 'closed?)))

(defmethod trivial-gray-streams:stream-write-byte ((stream writer) (integer integer))
  (declare (optimize speed (debug 1)))
  (check-type integer t/uint8)
  (with-typed-slots (((or null block-compressor) block-comp))
      stream
    (unless (block-compressor-write-byte block-comp integer)
      (%writer-compress-block stream)
      (%writer-init-next-block stream)
      (block-compressor-write-byte block-comp integer))))

(muffling
  (defmethod trivial-gray-streams:stream-write-sequence ((stream writer) (sequence sequence) (start integer) (end integer) &key &allow-other-keys)
    (declare (optimize speed (debug 1) (safety 1)))
    (with-typed-slots ((block-compressor block-comp))
        stream
      (loop with pos fixnum = 0
            with offset fixnum = 0
            with len fixnum = (length sequence)
            while (< offset len)
            do (setf pos (block-compressor-write-sequence block-comp sequence offset (- len offset)))
               (when (< pos len)
                 (%writer-compress-block stream)
                 (%writer-init-next-block stream))
               (incf offset pos)))
    sequence))

(defmethod %writer-init-next-block ((stream writer))
  (declare (optimize speed (debug 1)))
  (with-typed-slots (((or null block-compressor) block-comp)
                     (bit-writer bit-writer)
                     (t/int32 stream-block-size)
                     (array-pool int32-pool))
      stream
    (setf block-comp (make-block-compressor bit-writer stream-block-size int32-pool))))

(defmethod %writer-compress-block ((stream writer))
  (declare (optimize speed (debug 1)))
  (with-typed-slots (((or null block-compressor) block-comp)
                     (t/uint32 stream-crc))
      stream
    (unless (block-compressor-empty-p block-comp)
      (block-compressor-close-block block-comp)
      (setf stream-crc (coerce-to-uint32
                        (logxor (logior (ash stream-crc 1)
                                        (ash stream-crc -31))
                                (block-compressor-crc block-comp)))))))

(defmethod %writer-finish ((stream writer))
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((boolean closed?)
                     (stream output)
                     (bit-writer bit-writer)
                     (t/uint32 stream-crc)
                     ((or null block-compressor) block-comp))
      stream
    (unless closed?
      (unwind-protect
           (progn
             (%writer-compress-block stream)
             (bit-writer-write-bits bit-writer 24 +stream-end-marker-1+)
             (bit-writer-write-bits bit-writer 24 +stream-end-marker-2+)
             (bit-writer-write-int32 bit-writer stream-crc)
             (bit-writer-flush bit-writer)
             (finish-output output))
        (setf block-comp nil)
        (setf closed? t))))
  t)

(defmacro with-bzip2-writer ((val output-stream &key (level +default-compression-level+)) &body forms)
  `(let ((,val (make-writer ,output-stream :level ,level)))
     (unwind-protect
          (progn ,@forms)
       (close ,val))))
