;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; Copyright (C) 2015 Jaime Olivares
;;;; Copyright (c) 2011 Matthew Francis
;;;; Ported from the Java implementation by Matthew Francis:
;;;; https://github.com/MateuszBartosiewicz/bzip2.
;;;;
;;;; Ported by Remilia Scarlet from the C# implementation by Jamie Olivares:
;;;; http://github.com/jaime-olivares/bzip2
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-bzip2)

;;;;
;;;; An in-place, length restricted Canonical Huffman code length allocator.
;;;;
;;;; Based on the algorithm proposed by R.L.Milidiú, A.A.Pessoa and E.S.Laber in
;;;; "In-place Length-Restricted Prefix Coding"
;;;; (http://www-di.inf.puc-rio.br/~laber/public/spire98.ps) and incorporating
;;;; additional ideas from the implementation of shcodec
;;;; (http://webcenter.ru/~xander/) by Simakov.
;;;;

(define-typed-fn %huff-first ((t/int32-array array) (t/int32 the-i nodes-to-move))
    (t/int32)
  (declare (optimize speed (debug 1)))
  (let* ((i the-i)
         (len (length array))
         (limit i)
         (k (- (length array) 2)))
    (declare (type t/int32 len limit k i))

    (loop while (and (>= i nodes-to-move)
                     (> (mod (aref array i) len) limit))
          do (setf k i)
             (decf i (1+ (- limit i))))
    (setf i (max (1- nodes-to-move) i))

    (loop while (> k (1+ i))
          for tmp fixnum = (ash (+ i k) -1)
          if (> (mod (aref array tmp) len) limit) do
            (setf k tmp)
          else do
            (setf i tmp)
          finally (return k))))

(define-typed-fn %set-extended-parent-pointers ((t/int32-array array))
    (null)
  "Fills the code array with extended parent pointers."
  (declare (optimize speed (debug 1)))
  (let ((len (length array)))
    (incf (aref array 0) (aref array 1))
    (loop with head-node fixnum = 0
          with tail-node fixnum = 1
          with top-node fixnum = 2
          with tmp fixnum = 0
          while (< tail-node (1- len))
          do (setf tmp 0)
             (cond
               ((or (>= top-node len)
                    (< (aref array head-node) (aref array top-node)))
                (setf tmp (aref array head-node))
                (setf (aref array head-node) tail-node)
                (incf head-node))
               (t
                (setf tmp (aref array top-node))
                (incf top-node)))

             (cond
               ((or (>= top-node len)
                    (and (< head-node tail-node)
                         (< (aref array head-node) (aref array top-node))))
                (incf tmp (aref array head-node))
                (setf (aref array head-node) (+ tail-node len))
                (incf head-node))
               (t
                (incf tmp (aref array top-node))
                (incf top-node)))

             (setf (aref array tail-node) tmp)
             (incf tail-node)))
  nil)

(define-typed-fn %allocate-node-lengths ((t/int32-array array))
    (null)
  "A final allocation pass with no code length limit."
  (declare (optimize speed (debug 1)))
  (loop with first-node fixnum = (- (length array) 2)
        with next-node fixnum = (1- (length array))
        with current-depth fixnum = 1
        with available-nodes fixnum = 2
        with last-node fixnum = 0
        while (plusp available-nodes)
        do (setf last-node first-node)
           (setf first-node (%huff-first array (1- last-node) 0))

           (loop for i fixnum from (- available-nodes (- last-node first-node)) downto 1 do
             (setf (aref array next-node) current-depth)
             (decf next-node))

           (setf available-nodes (ash (- last-node first-node) 1))
           (incf current-depth))
  nil)

(define-typed-fn %allocate-node-lengths-with-relocation ((t/int32-array array) (t/int32 nodes-to-move insert-depth))
    (null)
  "A final allocation pass that relocates nodes in order to achieve a maximum
code length limit."
  (declare (optimize speed (debug 1)))
  (loop with first-node fixnum = (- (length array) 2)
        with next-node fixnum = (1- (length array))
        with current-depth fixnum = (if (= insert-depth 1) 2 1)
        with nodes-left-to-move fixnum = (if (= insert-depth 1) (- nodes-to-move 2) nodes-to-move)
        with available-nodes fixnum = (ash current-depth 1)
        with last-node fixnum = 0
        with offset fixnum = 0
        while (plusp available-nodes)
        do (setf last-node first-node)
           (setf first-node (if (<= first-node nodes-to-move)
                                first-node
                                (%huff-first array (1- last-node) nodes-to-move)))

           (setf offset 0)
           (cond
             ((>= current-depth insert-depth)
              (setf offset (min nodes-left-to-move (the fixnum (ash 1 (- current-depth insert-depth))))))
             ((= current-depth (1- insert-depth))
              (setf offset 1)
              (when (= (aref array first-node) last-node)
                (incf first-node))))

           (loop for i fixnum from (- available-nodes (+ (the fixnum (- last-node first-node)) offset)) downto 1 do
             (setf (aref array next-node) current-depth)
             (decf next-node))

           (decf nodes-left-to-move offset)
           (setf available-nodes (ash (+ (the fixnum (- last-node first-node)) offset) 1))
           (incf current-depth))
  nil)

(define-typed-fn allocate-huffman-code-lengths ((t/int32-array arr) (fixnum max-length))
    (null)
  "Allocates Canonical Huffman code lengths in place based on a sorted frequency array.

ARR should be a sorted array of symbol frequencies.  These will be modified
in-place into an array of canonical Huffman code lengths.  MAX-LENGTH is the
maximum code length.  This must be at least (CEILING (LOG (LENGTH ARR) 2))."
  (declare (optimize speed (debug 1)))
  (labels
      ((significant-bits (the-x)
         (declare (type fixnum the-x)
                  (optimize speed (debug 0) (safety 0) (compilation-speed 0)))
         (loop with n fixnum = 0
               with x fixnum = the-x
               while (plusp x) do
                 (setf x (ash x -1))
                 (incf n)
               finally (return n)))

       ;; Finds the number of nodes to relocate in order to achieve a given code
       ;; length limit.
       (find-nodes-to-relocate (array max-len)
         (declare (type t/int32-array array)
                  (t/int32 max-len)
                  (optimize speed (debug 0) (safety 0) (compilation-speed 0)))
         (loop with cur-node fixnum = (- (length array) 2)
               with cur-depth fixnum = 1
               with last-node fixnum = 0
               while (and (< cur-depth (1- max-len))
                          (> cur-node))
               do (setf cur-node (%huff-first array (1- cur-node) 0))
                  (incf cur-depth)
               finally (return cur-node))))
    (declare (inline significant-bits find-nodes-to-relocate))

    (case (length arr)
      (2
       (setf (aref arr 1) 1))
      (1
       (setf (aref arr 0) 1)
       (return-from allocate-huffman-code-lengths)))

    (%set-extended-parent-pointers arr)
    (let ((nodes-to-relocate (find-nodes-to-relocate arr max-length)))
      (if (>= (mod (aref arr 0) (length arr)) nodes-to-relocate)
          (%allocate-node-lengths arr)
          (let ((insert-depth (- max-length (significant-bits (1- nodes-to-relocate)))))
            (%allocate-node-lengths-with-relocation arr nodes-to-relocate insert-depth))))))
