;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; Copyright (C) 2015 Jaime Olivares
;;;; Copyright (c) 2011 Matthew Francis
;;;; Ported from the Java implementation by Matthew Francis:
;;;; https://github.com/MateuszBartosiewicz/bzip2.
;;;;
;;;; Ported by Remilia Scarlet from the C# implementation by Jamie Olivares:
;;;; http://github.com/jaime-olivares/bzip2
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-bzip2)

;;;;
;;;; BZip2-specific CRC32 Algorithm
;;;;

(defconst +crc32-lookup+
  (new-array-with
   t/uint32
   #(#x00000000 #x04C11DB7 #x09823B6E #x0D4326D9 #x130476DC #x17C56B6B #x1A864DB2 #x1E475005 #x2608EDB8
      #x22C9F00F #x2F8AD6D6 #x2B4BCB61 #x350C9B64 #x31CD86D3 #x3C8EA00A #x384FBDBD #x4C11DB70 #x48D0C6C7
      #x4593E01E #x4152FDA9 #x5F15ADAC #x5BD4B01B #x569796C2 #x52568B75 #x6A1936C8 #x6ED82B7F #x639B0DA6
      #x675A1011 #x791D4014 #x7DDC5DA3 #x709F7B7A #x745E66CD #x9823B6E0 #x9CE2AB57 #x91A18D8E #x95609039
      #x8B27C03C #x8FE6DD8B #x82A5FB52 #x8664E6E5 #xBE2B5B58 #xBAEA46EF #xB7A96036 #xB3687D81 #xAD2F2D84
      #xA9EE3033 #xA4AD16EA #xA06C0B5D #xD4326D90 #xD0F37027 #xDDB056FE #xD9714B49 #xC7361B4C #xC3F706FB
      #xCEB42022 #xCA753D95 #xF23A8028 #xF6FB9D9F #xFBB8BB46 #xFF79A6F1 #xE13EF6F4 #xE5FFEB43 #xE8BCCD9A
      #xEC7DD02D #x34867077 #x30476DC0 #x3D044B19 #x39C556AE #x278206AB #x23431B1C #x2E003DC5 #x2AC12072
      #x128E9DCF #x164F8078 #x1B0CA6A1 #x1FCDBB16 #x018AEB13 #x054BF6A4 #x0808D07D #x0CC9CDCA #x7897AB07
      #x7C56B6B0 #x71159069 #x75D48DDE #x6B93DDDB #x6F52C06C #x6211E6B5 #x66D0FB02 #x5E9F46BF #x5A5E5B08
      #x571D7DD1 #x53DC6066 #x4D9B3063 #x495A2DD4 #x44190B0D #x40D816BA #xACA5C697 #xA864DB20 #xA527FDF9
      #xA1E6E04E #xBFA1B04B #xBB60ADFC #xB6238B25 #xB2E29692 #x8AAD2B2F #x8E6C3698 #x832F1041 #x87EE0DF6
      #x99A95DF3 #x9D684044 #x902B669D #x94EA7B2A #xE0B41DE7 #xE4750050 #xE9362689 #xEDF73B3E #xF3B06B3B
      #xF771768C #xFA325055 #xFEF34DE2 #xC6BCF05F #xC27DEDE8 #xCF3ECB31 #xCBFFD686 #xD5B88683 #xD1799B34
      #xDC3ABDED #xD8FBA05A #x690CE0EE #x6DCDFD59 #x608EDB80 #x644FC637 #x7A089632 #x7EC98B85 #x738AAD5C
      #x774BB0EB #x4F040D56 #x4BC510E1 #x46863638 #x42472B8F #x5C007B8A #x58C1663D #x558240E4 #x51435D53
      #x251D3B9E #x21DC2629 #x2C9F00F0 #x285E1D47 #x36194D42 #x32D850F5 #x3F9B762C #x3B5A6B9B #x0315D626
      #x07D4CB91 #x0A97ED48 #x0E56F0FF #x1011A0FA #x14D0BD4D #x19939B94 #x1D528623 #xF12F560E #xF5EE4BB9
      #xF8AD6D60 #xFC6C70D7 #xE22B20D2 #xE6EA3D65 #xEBA91BBC #xEF68060B #xD727BBB6 #xD3E6A601 #xDEA580D8
      #xDA649D6F #xC423CD6A #xC0E2D0DD #xCDA1F604 #xC960EBB3 #xBD3E8D7E #xB9FF90C9 #xB4BCB610 #xB07DABA7
      #xAE3AFBA2 #xAAFBE615 #xA7B8C0CC #xA379DD7B #x9B3660C6 #x9FF77D71 #x92B45BA8 #x9675461F #x8832161A
      #x8CF30BAD #x81B02D74 #x857130C3 #x5D8A9099 #x594B8D2E #x5408ABF7 #x50C9B640 #x4E8EE645 #x4A4FFBF2
      #x470CDD2B #x43CDC09C #x7B827D21 #x7F436096 #x7200464F #x76C15BF8 #x68860BFD #x6C47164A #x61043093
      #x65C52D24 #x119B4BE9 #x155A565E #x18197087 #x1CD86D30 #x029F3D35 #x065E2082 #x0B1D065B #x0FDC1BEC
      #x3793A651 #x3352BBE6 #x3E119D3F #x3AD08088 #x2497D08D #x2056CD3A #x2D15EBE3 #x29D4F654 #xC5A92679
      #xC1683BCE #xCC2B1D17 #xC8EA00A0 #xD6AD50A5 #xD26C4D12 #xDF2F6BCB #xDBEE767C #xE3A1CBC1 #xE760D676
      #xEA23F0AF #xEEE2ED18 #xF0A5BD1D #xF464A0AA #xF9278673 #xFDE69BC4 #x89B8FD09 #x8D79E0BE #x803AC667
      #x84FBDBD0 #x9ABC8BD5 #x9E7D9662 #x933EB0BB #x97FFAD0C #xAFB010B1 #xAB710D06 #xA6322BDF #xA2F33668
      #xBCB4666D #xB8757BDA #xB5365D03 #xB1F740B4)))

(defstruct (bzip2-crc32 (:conc-name %crc-))
  (crc #xffffffff :type t/uint32)) ;; The current CRC

(define-typed-fn crc-crc ((bzip2-crc32 crc))
    (t/uint32 t)
  (declare (optimize speed (debug 1)))
  (coerce-to-uint32 (lognot (%crc-crc crc))))

(define-typed-fn crc-update ((bzip2-crc32 crc) (t/uint8 value))
    (null t)
  (declare (optimize speed (debug 1)))
  (setf (%crc-crc crc)
        (logxor (coerce-to-uint32 (ash (%crc-crc crc) 8))
                (aref +crc32-lookup+
                      (logand (logxor (ash (%crc-crc crc) -24) value) #xFF))))
  nil)

(define-typed-fn crc-update* ((bzip2-crc32 crc) (t/uint8 value) (fixnum count))
    (null)
  "Updates the CRC with a sequence of COUNT identical bytes."
  (declare (optimize speed (debug 1)))
  (loop repeat count do (crc-update crc value))
  nil)
