;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; Copyright (C) 2022 Jaime Olivares
;;;; Copyright (C) 2015 Jaime Olivares
;;;; Copyright (c) 2011 Matthew Francis
;;;; Ported from the Java implementation by Matthew Francis:
;;;; https://github.com/MateuszBartosiewicz/bzip2.
;;;;
;;;; Ported by Remilia Scarlet from the C# implementation by Jamie Olivares:
;;;; http://github.com/jaime-olivares/bzip2
;;;;
;;;; Modified by drone1400
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-bzip2)

(defgeneric block-compressor-empty-p (comp)
  (:documentation "Returns Tif any bytes have been written to the block, or NIL
otherwise."))

(defgeneric block-compressor-crc (comp)
  (:documentation "Returns the CRC of the completed block. Only valid after
calling BLOCK-COMPRESSOR-CLOSE-BLOCK."))

(defgeneric block-compressor-write-byte (comp value)
  (:documentation "Writes a byte to the block, accumulating to an RLE run where
possible.  Returns T on success, or NIL if the block is already full."))

(defgeneric block-compressor-write-sequence (comp value start end)
  (:documentation "Writes VALUE to the block.  This returns the first index of
VALUE that was not written, which may be zero if the block is already full."))

(defgeneric block-compressor-close-block (comp)
  (:documentation "Compresses and writes out the block."))

(defgeneric %block-compressor-write-run (comp value run-length)
  (:documentation "Writes an RLE run to the block array, updating the block CRC
and present values array as required."))

(defclass block-compressor ()
  ((output
    :type bit-writer
    :documentation "The stream to which compressed BZip2 data is written.")

   (crc
    :initform (make-bzip2-crc32)
    :type bzip2-crc32
    :documentation "CRC builder for the block.")

   (blk
    :type t/uint8-array
    :documentation "The RLE'd block data.")

   (block-length
    :initform 0
    :type t/int32
    :documentation "Current length of the data within the block array.")

   (block-length-limit
    :type t/int32
    :documentation "A limit beyond which new data will not be accepted into the block.")

   (block-values-present
    :initform (new-array 256 boolean nil)
    :type (simple-array boolean (256))
    :documentation "The values that are present within the RLE'd block data. For
each index, T if that value is present within the data, otherwise NIL.")

   (bwt-block
    :type t/int32-array
    :documentation "The Burrows Wheeler Transformed block data.")

   (rle-current-value
    :initform -1
    :type t/int32
    :documentation "The current RLE value being accumulated (undefined when
RLE-LENGTH is 0).")

   (rle-length
    :initform 0
    :type t/int32
    :documentation "The repeat count of the current RLE value.")

   (closed?
    :initform nil
    :type boolean)

   (int32-pool
    :type array-pool))

  (:documentation "Compresses and writes a single BZip2 block.  Use MAKE-BLOCK-COMPRESSOR to create
a new instance.

Block encoding consists of the following stages:

1. Run-Length Encoding[1]
2. Burrows Wheeler Transform (through DIV-SUF-SORT)
3. Write block header
4. Move To Front Transform (through HUFFMAN-STAGE-ENCODER)
5. Run-Length Encoding[2]  (through HUFFMAN-STAGE-ENCODER)
6. Create and write Huffman tables (through HUFFMAN-STAGE-ENCODER)
7. Huffman encode and write data (through HUFFMAN-STAGE-ENCODER)"))

(defun make-block-compressor (output block-size int32-pool)
  "Creates a new BLOCK-COMPRESSOR instance that will write compressed data to OUTPUT.

The BLOCK-SIZEparameter is the declared block size in bytes. Up to this many
bytes will be accepted into the block after run-length encoding is applied."
  (check-type output bit-writer)
  (check-type block-size t/int32)
  (check-type int32-pool array-pool)
  (let ((ret (make-instance 'block-compressor)))
    ;; One extra byte is added to allow for the block wrap applied when closing.
    (setf (slot-value ret 'output) output)
    (setf (slot-value ret 'int32-pool) int32-pool)
    (setf (slot-value ret 'blk) (new-array (1+ block-size) t/uint8))
    (setf (slot-value ret 'bwt-block) (array-pool-rent int32-pool (1+ block-size)))
    (fill (slot-value ret 'bwt-block) 0)

    ;; 5 bytes for one RLE run plus one byte.
    (setf (slot-value ret 'block-length-limit) (- block-size 6))

    ret))

(defmethod block-compressor-empty-p ((comp block-compressor))
  (declare (optimize speed (debug 1) (safety 1)))
  (with-typed-slots ((t/int32 block-length rle-length))
      comp
    (and (zerop block-length)
         (zerop rle-length))))

(defmethod block-compressor-crc ((comp block-compressor))
  (declare (optimize speed (debug 1) (safety 1)))
  (with-typed-slots ((bzip2-crc32 crc)) comp
    (crc-crc crc)))

(defmethod block-compressor-write-byte ((comp block-compressor) (value integer))
  (declare (optimize speed (debug 1)))
  (check-type value t/uint8)

  (with-typed-slots ((t/int32 block-length block-length-limit rle-length rle-current-value))
      comp
    (when (> block-length block-length-limit)
      ;; No more space on this block.
      (return-from block-compressor-write-byte nil))

    (cond
      ((zerop rle-length)
       (setf rle-current-value value)
       (setf rle-length 1))

      ((/= rle-current-value value)
       ;; This path commits us to write 6 bytes - one RLE run (5 bytes) plus one
       ;; extra.
       (%block-compressor-write-run comp (coerce-to-uint8 rle-current-value) rle-length)
       (setf rle-current-value value)
       (setf rle-length 1))

      (t
       (cond
         ((= rle-length 254)
          (%block-compressor-write-run comp (coerce-to-uint8 rle-current-value) 255)
          (setf rle-length 0))
         (t
          (incf rle-length))))))
  t)

(defmethod block-compressor-write-sequence ((comp block-compressor) (value vector) (start integer) (end integer))
  (declare (optimize speed (debug 1)))
  (muffling
    (when (< start 0)
      (error "START must be at least 0"))
    (when (< end start)
      (error "END cannot come before START"))
    (when (> end (length value))
      (error "Bad END for a sequence of ~d elements" (length value)))
    (loop for pos  from start below end
          for val across value
          while (block-compressor-write-byte comp val)
          finally (return pos))))

(defmethod block-compressor-write-sequence ((comp block-compressor) (value list) (start integer) (end integer))
  (declare (optimize speed (debug 1)))
  (muffling
    (when (< start 0)
      (error "START must be at least 0"))
    (when (< end start)
      (error "END cannot come before START"))
    (when (> end (length value))
      (error "Bad END for a sequence of ~d elements" (length value))))
  (loop for pos from start below end
        for val in value
        while (block-compressor-write-byte comp val)
        finally (return pos)))

(muffling
  (defmethod block-compressor-write-sequence ((comp block-compressor) (value sequence) (start integer) (end integer))
    (declare (optimize speed (debug 1)))
    (when (< start 0)
      (error "START must be at least 0"))
    (when (< end start)
      (error "END cannot come before START"))
    (when (> end (length value))
      (error "Bad END for a sequence of ~d elements" (length value)))
    (loop for pos from start below end
          for val = (elt value pos)
          while (block-compressor-write-byte comp val)
          finally (return pos))))

(defmethod block-compressor-close-block ((comp block-compressor))
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((t/int32 rle-length rle-current-value block-length)
                     ((simple-array boolean (256)) block-values-present)
                     (t/uint8-array blk)
                     (t/int32-array bwt-block)
                     (bit-writer output)
                     (array-pool int32-pool)
                     (bzip2-crc32 crc)
                     (boolean closed?))
      comp
    (labels
        ((write-symbol-map ()
           (let ((condensed-in-use (make-list 16 :initial-element nil))
                 (j 0)
                 (k 0))
             (declare (type list condensed-in-use)
                      (type t/int32 j k)
                      (dynamic-extent condensed-in-use))

             (dotimes (i 16)
               (setf j 0)
               (setf k (ash i 4))
               (loop while (< j 16) do
                 (when (aref block-values-present k)
                   (setf (nth i condensed-in-use) t))
                 (incf j)
                 (incf k)))

             (dolist (val condensed-in-use)
               (bit-writer-write-bool output val))

             (dotimes (i 16)
               (when (nth i condensed-in-use)
                 (setf j 0)
                 (setf k (* i 16))
                 (loop while (< j 16) do
                   (bit-writer-write-bool output (aref block-values-present k))
                   (incf j)
                   (incf k)))))))
      (declare (inline write-symbol-map))

      (when closed?
        (error "Attempted to close a block twice."))

      ;; If an RLE run is in progress, write it out.
      (when (plusp rle-length)
        (%block-compressor-write-run comp (coerce-to-uint8 rle-current-value) rle-length))

      ;; Apply a one byte block wrap required by the BWT implementation.
      (setf (aref blk block-length) (aref blk 0))

      ;; Perform the Burrows-Wheeler Transform.
      (let* ((transformer (%make-div-suf-sort :bytes blk
                                              :sa bwt-block
                                              :n block-length
                                              :int32-pool int32-pool))
             (bwt-start-point (dss-bwt transformer)))
        (declare (type t/int32 bwt-start-point))
        ;; Write out the block header.
        (bit-writer-write-bits output 24 +block-header-marker-1+)
        (bit-writer-write-bits output 24 +block-header-marker-2+)
        (bit-writer-write-int32 output (crc-crc crc))
        (bit-writer-write-bool output nil) ;; Randomised block flag. We never
                                           ;; create randomised blocks.
        (bit-writer-write-bits output 24 (coerce-to-uint32 bwt-start-point)))

      ;; Write out the symbol map.
      (write-symbol-map)

      ;; Perform the Move-To-Front Transform and run-length encoding[2] stages.
      (let ((mtf (make-instance 'mtf-and-rle-2-stage-encoder
                                :bwt-block bwt-block
                                :bwt-values-in-use block-values-present
                                :bwt-length block-length))
            (huffman nil))
        (mtf-rle-encoder-encode mtf)

        ;; Perform the Huffman Encoding stage and write out the encoded data.
        (setf huffman (make-instance 'huffman-stage-encoder
                                     :output output
                                     :mtf-block (mtf-rle-encoder-mtf-block mtf)
                                     :mtf-length (mtf-rle-encoder-mtf-length mtf)
                                     :mtf-alphabet-size (mtf-rle-encoder-mtf-alphabet-size mtf)
                                     :mtf-symbol-frequencies (mtf-rle-encoder-mtf-symbol-frequencies mtf)
                                     :int32-pool int32-pool))
        (huffman-stage-encoder-encode huffman))

      (array-pool-return int32-pool bwt-block)))
  nil)

(defmethod %block-compressor-write-run ((comp block-compressor) (value integer) (run-length integer))
  (declare (optimize speed (debug 1)))
  (check-type value t/uint8)
  (check-type run-length t/int32)
  (with-typed-slots (((simple-array boolean (256)) block-values-present)
                     (bzip2-crc32 crc)
                     (t/uint8-array blk)
                     (t/int32 block-length))
      comp
    (setf (aref block-values-present value) t)
    (crc-update* crc value run-length)

    (case run-length
      (1
       (setf (aref blk block-length) value)
       (incf block-length))

      (2
       (setf (aref blk block-length) value)
       (setf (aref blk (1+ block-length)) value)
       (incf block-length 2))

      (3
       (setf (aref blk block-length) value)
       (setf (aref blk (1+ block-length)) value)
       (setf (aref blk (+ block-length 2)) value)
       (incf block-length 3))

      (otherwise
       (decf run-length 4)
       (setf (aref block-values-present run-length) t)
       (setf (aref blk block-length) value)
       (setf (aref blk (1+ block-length)) value)
       (setf (aref blk (+ block-length 2)) value)
       (setf (aref blk (+ block-length 3)) value)
       (setf (aref blk (+ block-length 4)) (coerce-to-uint8 run-length))
       (incf block-length 5))))
  nil)
