;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; Copyright (C) 2022 Jaime Olivares
;;;; Copyright (C) 2015 Jaime Olivares
;;;; Copyright (c) 2011 Matthew Francis
;;;; Ported from the Java implementation by Matthew Francis:
;;;; https://github.com/MateuszBartosiewicz/bzip2.
;;;;
;;;; Ported by Remilia Scarlet from the C# implementation by Jamie Olivares:
;;;; http://github.com/jaime-olivares/bzip2
;;;;
;;;; Modified by drone1400
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-bzip2)

;;;;
;;;; BZip2-specific Bit Writer
;;;;

(defclass bit-writer ()
  ((io
    :initarg :io
    :type stream)

   (bit-buffer
    :initform 0
    :type t/uint32
    :documentation "A buffer of bits waiting to be written to the output stream.")

   (bit-count
    :initform 0
    :type t/int32
    :documentation "The number of bits currently buffered in BIT_BUFFER.")))

(define-typed-fn bit-writer-write-bool ((bit-writer writer) (boolean value))
    (null t)
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((stream io)
                     (t/uint32 bit-buffer)
                     (t/int32 bit-count))
      writer
    (incf bit-count)
    (logiorf bit-buffer (the t/uint32 (ash (if value 1 0) (- 32 bit-count))))
    (when (= bit-count 8)
      (write-byte (ash bit-buffer -24) io)
      (setf bit-buffer 0)
      (setf bit-count 0)))
  nil)

(define-typed-fn bit-writer-write-unary ((bit-writer writer) (fixnum value))
    (null t)
  (declare (optimize speed (debug 1)))
  (loop for val fixnum from value above 0 do
    (bit-writer-write-bool writer t))
  (bit-writer-write-bool writer nil))

(define-typed-fn bit-writer-write-bits ((bit-writer writer) (t/int32 count) ((or t/int32 t/uint32) value))
    (null t)
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((stream io)
                     (t/uint32 bit-buffer)
                     (t/int32 bit-count))
      writer
    (logiorf bit-buffer (coerce-to-uint32 (ash (logand (ash value (- 32 count)) #xFFFFFFFF)
                                               (- bit-count))))
    (incf bit-count count)

    (loop while (>= bit-count 8) do
      (write-byte (ash bit-buffer -24) io)
      (setf bit-buffer (coerce-to-uint32 (ash bit-buffer 8)))
      (decf bit-count 8)))
  nil)

(define-typed-fn bit-writer-write-int32 ((bit-writer writer) ((or t/int32 t/uint32) value))
    (null t)
  (declare (optimize speed (debug 1)))
  (bit-writer-write-bits writer 16 (logand (ash value -16) #xFFFF))
  (bit-writer-write-bits writer 16 (logand value #xFFFF)))

(define-typed-fn bit-writer-flush ((bit-writer writer))
    (null t)
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((t/int32 bit-count))
      writer
    (when (plusp bit-count)
      (bit-writer-write-bits writer (- 8 bit-count) 0))))
