;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Globals
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; TODO the string constants are likely SBCL-specific.
(defparameter *io-buffer-size*
  (cond
    ((string= (software-type) "Linux") (expt 2 24)) ;; Seems to work fast?
    ((or (string= (software-type) "Win32")
         (string= (software-type) "Win64"))
     65536) ;; Recommended by M$
    (t 65536))
  "The size of the buffer to use when using CL-SDM's I/O functions.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Common Functions/Macros, and Compatibility Utilities
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro gc (&optional full)
  "Performs garbage collection.  If FULL is truthy, and the Lisp implementation
supports it, a full collection is done."
  #+ccl (declare (ignore full))
  #+sbcl `(sb-ext:gc :full ,full)
  #+ccl `(ccl:gc)
  #+clisp `(ext:gc ,full)
  #+ecl `(sys:gc ,(if full t nil)))

(defun exit (&optional (code 0) #+sbcl (abort? nil))
  "Quits the Lisp instance."
  #+sbcl  (sb-ext:exit :code code :abort abort?)
  #+ecl   (ext:quit code)
  #+clisp (ext:quit code)
  #+ccl   (ccl::quit code))

(declaim (inline xor))
(defun xor (form1 form2)
  "Returns T if 'FORM1 xor FORM2' is logically true, or NIL otherwise."
  (and (not (and form1 form2))
       (not (and (not form1) (not form2)))))

(eval-when (:compile-toplevel :load-toplevel)
  (defmacro with-gensyms ((&rest names) &body body)
    "Taken from Practical Common Lisp by Peter Seibel.  Calls GENSYM on each
name in NAMES, constructing a new symbol name based on the name."
    `(let ,(loop for name in names collect `(,name (gensym ,(string name))))
       ,@body)))

(defmacro binding-multiple-values (value-specs &body forms)
  "Convenience macro for embedding multiple MULTIPLE-VALUE-BIND calls.
  VALUE-SPECS is a list where the first element is a list of symbols and the
  second element is the form that generates the values to bind to the symbols.
  These bindings can then be used in FORMS.

Example:
(binding-multiple-values (((a b c) (values 1 2 3))
                          ((foo bar baz) (values 4 5 6)))
  (print (list a b c foo bar baz)))

is equivalent to

(multiple-value-bind (a b c)
    (values 1 2 3)
  (multiple-value-bind (foo bar baz)
      (values 4 5 6)
    (print (list a b c foo bar baz))))"
  (let ((ret ()))
    (loop for spec in value-specs
          for i from 1 do
            (let ((mvb-block `(multiple-value-bind ,(car spec) ,(cadr spec))))
              (when (= i (length value-specs))
                (setf mvb-block (append mvb-block forms)))

              (if (> i 1)
                  (setf ret (append ret (list mvb-block)))
                  (setf ret (append ret mvb-block)))))
    ret))

(declaim (ftype (function (hash-table t) boolean) hash-table-contains-p)
         (inline hash-table-contains-p))
(defun hash-table-contains-p (table key)
  "Returns T if TABLE contains KEY, or NIL otherwise."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (nth-value 1 (gethash key table)))

(defmacro hash-let (table bindings &body forms)
  "Binds multiple hash values, then executes FORMS.

BINDINGS should be a list of binding definitions, where each definition is a
list in the form (VAR KEY &OPTIONAL FOUND).  When FOUND is present, it is set to
T if the key was found in the table, or NIL otherwise."
  (let ((ret ()))
    (dolist (binding (reverse bindings))
      (destructuring-bind (var key &optional found)
          binding
        (if (null ret)
            (setf ret (append (list 'multiple-value-bind (if found (list var found) (list var))
                                    (list 'gethash key table))
                              forms))
            (setf ret (list 'multiple-value-bind (if found (list var found) (list var))
                            (list 'gethash key table)
                            ret)))))
    ret))

(defun hash-try-add (hash key value)
  "Attempts to add VALUE to the hash table under KEY.  If KEY already exists in
the hash table, this does nothing.  Returns T on success, or NIL otherwise."
  (declare (type hash-table hash)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (multiple-value-bind (old-val found)
      (gethash key hash)
    (declare (ignore old-val))
    (unless found
      (setf (gethash key hash) value))
    (not found)))

;; Taken from the Alexandria library
(defun %reevaluate-constant (name value test)
  (if (not (boundp name))
      value
      (let ((old (symbol-value name))
            (new value))
        (if (not (constantp name))
            (prog1 new
              (cerror "Try to redefine the variable as a constant."
                      "~@<~S is an already bound non-constant variable ~
                       whose value is ~S.~:@>" name old))
            (if (funcall test old new)
                old
                (restart-case
                    (error "~@<~S is an already defined constant whose value ~
                              ~S is not equal to the provided initial value ~S ~
                              under ~S.~:@>" name old new test)
                  (ignore ()
                    :report "Retain the current value."
                    old)
                  (continue ()
                    :report "Try to redefine the constant."
                    new)))))))

(defmacro defconst (name value &key (test nil test-supplied-p) documentation)
  "Defines a constant.  If TEST is supplied, it will be used to test for
equality.  Otherwise, an appropriate function will be chosen based on the type
of VALUE.  This is effectively a shorthand way to call Alexandria's
DEFINE-CONSTANT."
  `(defconstant ,name (%reevaluate-constant ',name ,value ,(if test-supplied-p
                                                               test
                                                               (typecase value
                                                                 (number ''eql)
                                                                 (string ''equal)
                                                                 (otherwise ''equalp))))
     ,@(when documentation `(,documentation))))

(defmacro defining-consts (&body forms)
  "Defines a set of constants.  Their type will be detected and the appropriate
test will be passed to DEFCONST at expansion time.  Each takes the same
arguments as DEFCONST.

This is shorthand for multiple calls to DEFCONST."
  (loop for form in forms
        collect
        (let ((sym (nth 0 form))
              (val (nth 1 form)))
          (unless (symbolp sym)
            (error "Invalid symbol name"))

          (list 'defconst sym val
                :test (typecase val
                        (number ''eql)
                        (string ''equal)
                        (otherwise ''equalp))))
          into ret
        finally (return (append (list 'progn) ret))))

(declaim (ftype (function (string &key (:start fixnum) (:end (or null fixnum)) (:radix fixnum)) (or null integer))
                parse-integer?)
         (inline parse-integer?))
(defun parse-integer? (string &key (start 0) end (radix 10))
  "Attempts to parse an integer out of STRING.  This is exactly like
PARSE-INTEGER, except that on failure it returns NIL instead of raising a
condition."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0) (space 0)))
  (handler-bind
      ((parse-error (lambda (err)
                      (declare (ignore err))
                      (return-from parse-integer?))))
    (parse-integer string :start start :end end :radix radix)))


(defmacro define-typed-fn (name (&rest arg-list) (ret-type &optional inline) &body forms)
  "Declares a function named NAME that has typed arguments.

The arguments should be list that contains one of two things: symbols, or lists
of 2+ elements where the first element is the type and the rest of the elements
are arguments of that type, in the order they appear.  If the argument
definition is a bare symbol, then its typed as T.

This will automatically insert a DECLAIM regarding the arguments before the
DEFUN.

INLINE is evaluated only at macro expansion time.  If INLINE is equal to :NO,
then the function is explicitly declaimed as NOTINLINE.  If INLINE is otherwise
truthy, it is explicitly declaimed as INLINE.  Otherwise, no INLINE or NOTINLINE
declaration is made.

There are currently a few limitations to DEFINE-TYPED-FN.  If any of
these are too limiting, then you may want to use a normal (DECLAIM (FTYPE...
and a (DEFUN yourself until they are supported:

  * If you use &OPTIONAL or &REST, then the symbols after that can be
    typed, but cannot specify a default value (so if you want them
    typed, you probably want to specify the type as (OR NULL <type>).
  * &KEY is not supported.
  * &AUX is not supported."

  (loop for arg-def in arg-list
        with types = ()
        with args = ()
        do (typecase arg-def
             (list
              (cond
                ((>= (length arg-def) 2)
                 (dolist (arg (cdr arg-def))
                   (push (car arg-def) types)
                   (push arg args)))
                ((= (length arg-def) 1)
                 (push (car arg-def) args)
                 (push T types))
                (t (error "Invalid argument definition: ~a

An argument definition should take the form NAME, or (NAME TYPE)."
                          arg-def))))

             (symbol
              (push arg-def args)
              (cond
                ((or (eq arg-def '&optional)
                     (eq arg-def '&rest))
                 (push arg-def types))

                ((or (eq arg-def '&key)
                     (eq arg-def '&aux))
                 (error "DEFINE-TYPED-FN does not currently support ~a" arg-def))

                (t (push t types))))

             (otherwise
              (error "Invalid slot definition: ~a

An argument definition should take the form NAME, or (NAME TYPE)."
                     arg-def)))

        finally
           (return `(progn
                      ,(cond
                         ((eq inline :no)
                         `(declaim (ftype (function ,(reverse types) ,ret-type) ,name)
                                   (notinline ,name)))

                         ((not inline)
                          `(declaim (ftype (function ,(reverse types) ,ret-type) ,name)))

                         (t
                          `(declaim (ftype (function ,(reverse types) ,ret-type) ,name)
                                    (inline ,name))))
                      (defun ,name ,(reverse args)
                        ,@forms)))))

(defmacro with-typed-slots ((&rest slot-defs) obj &body forms)
  "This is similar to WITH-SLOTS, but allows you to specify types for the
accessors as well.  Some Lisp implementations do not record this information, so
this lets you DECLARE types for all the slot accessors in a simple way.

The SLOT-DEFS should be list that contains one of two things:

1. Symbols.  These are the slot names, and will be typed as T.
2. Lists of 2+ elements.  In this case the first element is always the type.
   The rest of the elements are slot names of that type, or lists of exactly two
   elements where the first element is the locally bound name for the slot and
   the second element is the actual slot name."
  (loop for slot-def in slot-defs
        with types = ()
        with slots = ()
        do (typecase slot-def
             (list
              (cond
                ((>= (length slot-def) 2)
                 (dolist (slot (cdr slot-def))
                   (etypecase slot
                     (list
                      (unless (= (length slot) 2)
                        (error "Bad WITH-TYPED-SLOTS syntax: slot names should either be symbols, or a list in the form (VARIABLE-NAME SLOT-NAME)."))
                      (push (list 'type (car slot-def) (car slot)) types)
                      (push slot slots))

                     (symbol
                      (push (list 'type (car slot-def) slot) types)
                      (push slot slots)))))

                ;; Single symbol in a list is the same as just a bare symbol,
                ;; except we store T as its type.
                ((= (length slot-def) 1)
                 (push (car slot-def) slots)
                 (push (list T (car slot-def)) types))

                (t (error "Invalid slot definition: ~a

A slot definition should take the form NAME, or (NAME TYPE)."
                          slot-def))))

             (symbol
              (push slot-def slots))
             (otherwise
              (error "Invalid slot definition: ~a

A slot definition should take the form NAME, or (NAME TYPE)." slot-def)))
        finally (return `(with-slots ,slots ,obj
                           (declare ,@types)
                           ,@forms))))

(trivial-indent:define-indentation with-typed-slots (6 4 &body))

(defmacro new-vector (type)
  "Convenience macro that creates a new, empty, adjustable vector of the given
type.  TYPE may be quoted or unquoted."
  (let ((actual-type (if (and (listp type)
                              (eq (car type) 'quote))
                         type
                         (list 'quote type))))
    `(make-array 0 :adjustable t :fill-pointer 0 :element-type ,actual-type)))

(defmacro new-array (size type &optional (initial-element 0))
  "Convenience macro that creates a new, non-adjustable array of the given size
and element type, and initializes it with the given element.  TYPE may be quoted
or unquoted."
  (let ((actual-type (if (and (listp type)
                              (eq (car type) 'quote))
                         type
                         (list 'quote type))))
    `(make-array ,size :element-type ,actual-type :initial-element ,initial-element)))

(defmacro new-array-with (type initial-contents)
  "Convenience macro that creates a new non-adjustable array of the given size
and element type, and initializes it with the given contents.  TYPE may be
quoted or unquoted."
  (let ((actual-type (if (and (listp type)
                              (eq (car type) 'quote))
                         type
                         (list 'quote type))))
    `(make-array (length ,initial-contents) :element-type ,actual-type :initial-contents ,initial-contents)))

(defmacro new-array-with-ctor (size type constructor-fn &rest ctor-args)
  (let ((actual-type (if (and (listp type)
                              (eq (car type) 'quote))
                         type
                         (list 'quote type))))
    `(muffling
       (make-array ,size :element-type ,actual-type
                         :initial-contents (loop repeat ,size
                                                 collect ,(if ctor-args
                                                              `(funcall (function ,constructor-fn) ,@ctor-args)
                                                              `(,constructor-fn)))))))

(defmacro dx-let ((&rest bindings) &body forms)
  "Similar to a LET* block, except that bindings that start with :DX are
automatically declared DYNAMIC-EXTENT.

Example:

(dx-let ((:dx (dx-var (some-fun 69)))
         (not-dx-var 42))
  (declare (type fixnum not-dx-var)
           (type (or null double-float) dx-var))
  (print dx-var)
  (print not-dx-var))

This example would declare the variable DX-VAR to be DYNAMIC-EXTENT and
automatically move its initialization value to the proper place.  NOT-DX-VAR is
the same as in a normal LET*

** IMPORTANT NOTE **
When mixing DYNAMIC-EXTENT and non-DYNAMIC-EXTENT bindings, the
non-DYNAMIC-EXTENT bindings are always moved to the top of the list of bindings.

** IMPORTANT NOTE #2 **
If you decide to DECLARE the types of the DYNAMIC-EXTENT bindings, you should
remember that they are initially set to NIL.  This is why DX-VAR is declared
as (OR NULL DOUBLE-FLOAT) in the example above.

This macro is considered EXPERIMENTAL and may still produce unexpected results.
It's suggested that you expand the macros and examine them to be sure it does
what you want."
  (labels ((get-dx-binding (bind)
             (if (listp bind)
                 (if (= (length bind) 2)
                     (if (symbolp (car bind))
                         (car bind)
                         (error "Invalid binding: ~a" bind))
                     (error "Invalid binding: ~a" bind))
                 (if (symbolp bind)
                     (error "Invalid DX binding: ~a" bind)
                     (error "Invalid binding: ~a" bind))))

           (get-val (bind)
             (if (listp bind)
                 (if (= (length bind) 2)
                     (if (symbolp (car bind))
                         (cadr bind)
                         (error "Invalid binding: ~a" bind))
                     (error "Invalid binding: ~a" bind))
                 (if (symbolp bind) nil))))
    (loop for binding in bindings
          if (and (listp binding)
                  (= (length binding) 2)
                  (eq (car binding) :dx))
            collect (get-dx-binding (cadr binding)) into dx-names
          else
            collect binding into normal-bindings

          when (and (listp binding)
                    (= (length binding) 2)
                    (eq (car binding) :dx))
            collect (get-val (cadr binding)) into dx-init-vals

          finally
             (return
               (let ((inits (loop for val in dx-init-vals
                                  for i from 0
                                  collect (when val
                                            `(setf ,(nth i dx-names) ,val))
                                    into inits
                                  finally (return (remove-if #'null inits))))
                     (dx-bindings (loop for name in dx-names collecting (list name nil))))

                 `(let* (,@normal-bindings
                         ,@dx-bindings)
                    ,(if (and (listp (car forms))
                              (eq (caar forms) 'cl:declare))
                         (prog1
                             `(declare ,(cadar forms)
                                       (dynamic-extent ,@dx-names))
                           (setf forms (subseq forms 1)))
                         `(declare (dynamic-extent ,@dx-names)))
                    ,@inits
                    ,@forms))))))

(defun find-program-in-path (program &optional (path-env "PATH"))
  "Loops through all of the paths found in PATH-ENV (which defaults to the
  environment variable \"PATH\"), searching each one in order for a program
  named PROGRAM.  This returns the path to the first program found, or NIL."
  (declare (type string program path-env))
  (loop named outer
        for path in (uiop:getenv-pathnames path-env)
        do (dolist (file (uiop:directory-files path))
             (when (string= (pathname-name file) program)
               (return-from outer file)))))

(defmacro with-bound-slot ((thing slot var) &body forms)
  "When the slot SLOT is bound in THING, its value is bound to VAR and then
FORMS are executed.  If the slot is not bound, this does nothing.  SLOT can be
quoted or unquoted."
  (let ((slot-sym (if (listp slot)
                      slot
                      (list 'quote slot))))
    `(when (slot-boundp ,thing ,slot-sym)
       (let ((,var (slot-value ,thing ,slot-sym)))
         (declare (ignorable ,var))
         ,@forms))))

(defmacro if-slot-bound ((thing slot var) bound-form &optional not-bound-form)
  "If the slot SLOT is bound in THING, its value is bound to VAR and then
BOUND-FORM is executed.  If the slot is not bound, then VAR remains unbound and
NOT-BOUND-FORM is executed."
  (let ((slot-sym (if (listp slot)
                      slot
                      (list 'quote slot))))
    `(let (,var)
       (declare (ignorable ,var))
       (if (slot-boundp ,thing ,slot-sym)
           (progn
             (setf ,var (slot-value ,thing ,slot-sym))
             ,bound-form)
           ,not-bound-form))))

(trivial-indent:define-indentation if-slot-bound (2 4 4))

(defun create-empty-file (filename size &key (if-exists :error))
  "Creates an empty file at FILENAME of the given size.  IF-EXISTS is the same
as for OPEN and defaults to :ERROR."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type size fixnum)
  (check-type filename (or pathname string))
  (with-open-file (out filename :direction :output
                                :element-type '(unsigned-byte 8)
                                :if-exists if-exists)
    (unless (zerop size)
      (file-position out (1- size))
      (write-byte 0 out)))
  t)

(defmacro muffling (&body forms)
  "On SBCL, this muffles compiler notes within FORMS.  On other implementations,
this just executes forms without any additional declarations."
  `(locally
       (declare #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
     ,@forms))

(defmacro define-simple-error (name parent (&rest additional-inherits) &body slots)
  "Defines a condition named NAME that inherits from SIMPLE-ERROR, as well as a
macro with the same name that can be used to raised conditions of the new type.

If PARENT is NIL, then the new condition inherits directly from SIMPLE-ERROR.
Otherwise it inherits from PARENT plus any additional classes specified in
ADDITIONAL-INHERITS.

SLOTS is passed as-is to the DEFINE-CONDITION call where slots would normally be
defined.

If PARENT is non-NIL, then the macro that is generated will ensure that whatever
you pass to it is a subtype of that type.  If it is not provided, then it checks
that anything passed to it is a subclass of the new condition that gets
defined."
  (let ((subtype (or parent name))
        (real-slots (if slots slots '(nil))))
    (declare (ignorable subtype))
    `(progn
       (define-condition ,name ,(if additional-inherits
                                    (append (list (or parent 'simple-error))
                                            additional-inherits)
                                    (if parent
                                        (list parent)
                                        '(simple-error)))
         ,@real-slots)

       (defmacro ,name ((&optional (type '',name)) msg &body fmt-args)
         (let ((type (if (listp type) type (list 'quote type))))
           #-ccl
           (unless (subtypep (cadr type) ,(if (listp subtype)
                                              subtype
                                              (list 'quote subtype)))
             (error "Not a subtype of ~a: ~a"
                    ,(if (listp subtype)
                         subtype
                         (list 'quote subtype))
                    type))
           `(error ,type :format-control ,msg :format-arguments (list ,@fmt-args)))))))

(defmacro when-var ((var form) &body forms)
  "Binds the result of FORM to VAR.  If VAR is truthy, FORMS are executed.

This returns two values: if FORMS are executed, then it returns the result of
FORMS and T.  If FORMS are not executed, it returns NIL and NIL."
  `(let ((,var ,form))
     (if ,var
         (values (progn ,@forms) t)
         (values nil nil))))

(defmacro with-interrupts-disabled (&body forms)
  "Disables interrupts (if applicable to the running implementation), then
executes FORMS.  The difference betwen this and WITHOUT-INTERRUPTS is how things
are disabled.  You may want to MACROEXPAND-1 your form to see the difference and
select the appropriate code."
  `(let #+sbcl ((sb-sys:*interrupts-enabled* nil)
                (sb-kernel:*gc-inhibit* t))
     #-sbcl ()
     ,@forms))

(defmacro without-interrupts (&body forms)
  "Disables interrupts (if applicable to the running implementation), then
executes FORMS.  The difference betwen this and WITH-INTERRUPTS-DISABLED is how
things are disabled.  You may want to MACROEXPAND-1 your form to see the
difference and select the appropriate code."
  #+sbcl `(sb-sys:without-interrupts ,@forms)
  #-sbcl `(progn ,@forms))

(defun merge-pathnames-as-dir (specified &optional (default *default-pathname-defaults*))
  "Merges DEFAULT and SPECIFIED together, ensuring the result is a directory pathname."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (uiop:ensure-directory-pathname
   (uiop:merge-pathnames*
    specified
    default)))

(defun merge-paths (components &key (default *default-pathname-defaults*) return-as-dir)
  "Merges COMPONENTS together as one long pathname."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type components list)

  (case (length components)
    (0
     nil)
    (1
     (let ((ret (uiop:merge-pathnames* (car components) default)))
       (if return-as-dir
           (uiop:ensure-directory-pathname ret)
           ret)))
    (otherwise
     (loop with ret = nil
           with len fixnum = (length components)
           for comp in components
           for idx fixnum from 0
           if (zerop idx) do
             (setf ret (merge-pathnames-as-dir comp default))
           else if (= idx (1- len)) do
             (setf ret (uiop:merge-pathnames* comp ret))
           else do
             (setf ret (merge-pathnames-as-dir comp ret))
           finally
              (return (if return-as-dir
                          (uiop:ensure-directory-pathname ret)
                          ret))))))

(declaim (ftype (function (vector) (values T boolean)) vec-last)
         (inline vec-last))
(defun vec-last (vec)
  "Returns two values, depending on the size of VEC:

* If VEC has at least one element, this returns the last element in VEC and T.
* If VEC has no elements, this returns NIL and NIL."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (if (zerop (length vec))
      (values nil nil)
      (values (muffling (elt vec (1- (length vec))))
              t)))

(declaim (ftype (function (vector) (values T boolean)) vec-first)
         (inline vec-first))
(defun vec-first (vec)
  "Returns two values, depending on the size of VEC:

* If VEC has at least one element, this returns the first element in VEC and T.
* If VEC has no elements, this returns NIL and NIL."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (if (zerop (length vec))
      (values nil nil)
      (values (muffling (elt vec 0))
              t)))

;;;
;;; Internal macros for working with SB-CLTL2 to get variable information
;;;

#+sbcl
(defmacro %var-is-type (env var the-type)
  (with-gensyms (type)
    `(let ((,type (multiple-value-list (sb-cltl2:variable-information ,var ,env))))
       (and (listp ,type)
            (eq (nth 0 ,type) :lexical)
            (>= (length ,type) 3)
            (subtypep (cdr (caaddr ,type)) ,(list 'quote the-type))))))
