;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/clocks)

;;;;
;;;; Clocks - Unix Support
;;;;
;;;; Originally from monotonic-clock by death, modified by Remilia Scarlet.
;;;;

(eval-when (:compile-toplevel :load-toplevel)
  (deftype t/mode ()
    "A T/MODE defines the modes (clock types) available for reading.  These clocks
are available on all POSIX systems:

* :DEFAULT - This uses the CLOCK_MONOTONIC clock.
* :REALTIME - This uses the CLOCK_REALTIME clock.

These clocks are available only on Linux:

* :REALTIME-COARSE - This uses the CLOCK_REALTIME_COARSE clock.
* :COARSE - This uses the CLOCK_MONOTONIC_COARSE clock.
* :BOOTTIME - This uses the CLOCK_BOOTTIME clock.
* :PROCESS-CPUTIME - This uses the CLOCK_PROCESS_CPUTIME_ID clock.
* :THREAD-CPUTIME - This uses the CLOCK_THREAD_CPUTIME_ID clock.
* :RAW - This uses the CLOCK_MONOTONIC_RAW clock."
    '(member
      :default
      :realtime

      #+linux :realtime-coarse
      #+linux :coarse
      #+linux :boottime
      #+linux :process-cputime
      #+linux :thread-cputime
      #+linux :raw))

  (define-typed-fn t/mode->clock ((t/mode mode))
      (t t)
    (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0) (space 0)))
    (ecase mode
      (:default cl-sdm/clocks-lib:clock-monotonic)
      (:realtime cl-sdm/clocks-lib:clock-realtime)
      #+linux (:realtime-coarse cl-sdm/clocks-lib:clock-realtime-coarse)
      #+linux (:coarse cl-sdm/clocks-lib:clock-monotonic-coarse)
      #+linux (:boottime cl-sdm/clocks-lib:clock-boottime)
      #+linux (:process-cputime cl-sdm/clocks-lib:clock-process-cputime-id)
      #+linux (:thread-cputime cl-sdm/clocks-lib:clock-thread-cputime-id)
      #+linux (:raw cl-sdm/clocks-lib:clock-monotonic-raw)))

  (declaim (ftype (function (&optional t/mode) (values t/int64 t/int64)) clock-resolution))
  (defun clock-resolution (&optional (mode :default))
    "Returns the precision of the specified T/MODE.  The first value is the seconds,
the second value is the nanoseconds."
    (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0) (space 0))
             #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
    (check-type mode t/mode)
    (cffi:with-foreign-object (res '(:struct cl-sdm/clocks-lib:timespec))
      (unless (zerop (the fixnum (cl-sdm/clocks-lib:clock-getres (t/mode->clock mode) res)))
        (clock-error () "clock_getres returned ~a" cl-sdm/clocks-lib:*errno*))
      (values (cffi:foreign-slot-value res
                                       '(:struct cl-sdm/clocks-lib:timespec)
                                       'cl-sdm/clocks-lib:sec)
              (cffi:foreign-slot-value res
                                       '(:struct cl-sdm/clocks-lib:timespec)
                                       'cl-sdm/clocks-lib:nsec)))))

(defining-consts
  (+monotonic-time-units-per-second+ (/ 1000000000 (nth-value 1 (clock-resolution :default))))
  (+realtime-time-units-per-second+ (/ 1000000000 (nth-value 1 (clock-resolution :realtime))))
  #+linux (+realtime-time-units-per-second/coarse+ (/ 1000000000 (nth-value 1 (clock-resolution :realtime-coarse))))
  #+linux (+monotonic-time-units-per-second/coarse+ (/ 1000000000 (nth-value 1 (clock-resolution :coarse))))
  #+linux (+bootime-time-units-per-second+ (/ 1000000000 (nth-value 1 (clock-resolution :boottime))))
  #+linux (+process-cputime-time-units-per-second+ (/ 1000000000 (nth-value 1 (clock-resolution :process-cputime))))
  #+linux (+thread-cputime-time-units-per-second+ (/ 1000000000 (nth-value 1 (clock-resolution :thread-cputime))))
  #+linux (+monotonic-time-units-per-second/raw+ (/ 1000000000 (nth-value 1 (clock-resolution :raw)))))

(define-typed-fn %clock-get-time (clock)
    ((values t/int64 t/int64) t)
  "Calls the low-level C function CLOCK-GETTIME and returns its values.  The first
value is the sec value, the second is the nsec."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0) (space 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (cffi:with-foreign-object (tp '(:struct cl-sdm/clocks-lib:timespec) 1)
    (unless (zerop (the fixnum (cl-sdm/clocks-lib:clock-gettime clock tp)))
      (clock-error () "clock_gettime returned ~a" cl-sdm/clocks-lib:*errno*))

    (values (cffi:foreign-slot-value tp
                                     '(:struct cl-sdm/clocks-lib:timespec)
                                     'cl-sdm/clocks-lib:sec)
            (cffi:foreign-slot-value tp
                                     '(:struct cl-sdm/clocks-lib:timespec)
                                     'cl-sdm/clocks-lib:nsec))))

(define-typed-fn clock-time-units-per-second ((t/mode mode))
    (t/uint32 t)
  "Return the number of time units in one second for the given mode."
  (declare (optimize speed (debug 1) (safety 0) (compilation-speed 0) (space 0)))
  (ecase mode
    (:default +monotonic-time-units-per-second+)
    (:realtime +realtime-time-units-per-second+)
    #+linux (:realtime-coarse +realtime-time-units-per-second/coarse+)
    #+linux (:coarse +monotonic-time-units-per-second/coarse+)
    #+linux (:boottime +bootime-time-units-per-second+)
    #+linux (:process-cputime +process-cputime-time-units-per-second+)
    #+linux (:thread-cputime +thread-cputime-time-units-per-second+)
    #+linux (:raw +monotonic-time-units-per-second/raw+)))

(declaim (ftype (function (&optional t/mode) t/int64) clock-now)
         (inline clock-now))
(defun clock-now (&optional (mode :default))
  "Returns the current time in clock time units.  MODE must be a T/MODE.  The
default is to return a monotonic time."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0) (space 0)))
  (multiple-value-bind (sec nsec)
      (%clock-get-time (t/mode->clock mode))
    (muffling
      (+ (* sec (clock-time-units-per-second mode)) nsec))))

(declaim (ftype (function (&optional t/mode) t/int64) clock-now/ms)
         (inline clock-now/ms))
(defun clock-now/ms (&optional (mode :default))
  "Return the current time in milliseconds.  MODE must be a T/MODE.  The default is
to return a monotonic time."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0) (space 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (nth-value 0 (floor (clock-now mode) (floor (clock-time-units-per-second mode) 1000))))
