;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

(define-condition not-on-byte-error (simple-error)
  ())

(defstruct (bit-reader (:constructor %make-bit-reader)
                       (:conc-name %bit-reader-))
  (strm nil)
  (byte nil :type (or null t/uint8))
  (bit-pos 0 :type t/uint8))

(defun make-bit-reader (stream)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type stream stream)
  (%make-bit-reader :strm stream :byte (read-byte stream)))

(defun bit-reader-rewind (reader)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type reader bit-reader)
  (setf (bit-reader-pos reader) 0))

(defun bit-reader-reinitialize (reader stream)
  "Reinitializes READER with a new STREAM.  This completely resets READER.  This
will always call READ-BYTE exactly once to populate the initial BIT-READER-BYTE.
Returns the BIT-READER."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type reader bit-reader)
  (check-type stream stream)
  (setf (%bit-reader-bit-pos reader) 0)
  (setf (%bit-reader-strm reader) stream)
  (setf (%bit-reader-byte reader) (read-byte stream))
  reader)

(defun bit-reader-close (reader &key abort)
  "Closes the underlying stream in READER.  The ABORT parameter is the same as
for CLOSE."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type reader bit-reader)
  (close (%bit-reader-strm reader) :abort abort))

(define-typed-fn bit-reader-bit-pos ((bit-reader stream))
    (t/uint8)
  "Returns the current position (0-7, inclusive) in the current byte of the
BIT-READER."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (%bit-reader-bit-pos stream))

(define-typed-fn bit-reader-cur-byte ((bit-reader stream))
    ((or null t/uint8))
  "Returns the current byte of the BIT-READER, or NIL if there are no bytes
left."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (%bit-reader-byte stream))

(define-typed-fn bit-reader-pos ((bit-reader stream))
    (fixnum)
  "Returns the current byte position of the BIT-READER's underlying stream."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (1- (file-position (%bit-reader-strm stream))))

(define-typed-fn (setf bit-reader-pos) ((fixnum value) (bit-reader stream))
    (fixnum)
  "Sets the current byte position of the BIT-READER's underlying stream."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (file-position (%bit-reader-strm stream) value)
  (setf (%bit-reader-bit-pos stream) 0)
  (setf (%bit-reader-byte stream) (read-byte (%bit-reader-strm stream)))
  (1- (file-position (%bit-reader-strm stream))))

(define-typed-fn (setf bit-reader-cur-byte) ((t/uint8 value) (bit-reader stream))
    (t/uint8)
  "Sets what STREAM considers the last byte read.  This does not affect the
underlying stream."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (setf (%bit-reader-byte stream) value))

(declaim (ftype (function (bit-reader t/uint32 &optional T T) T) bit-reader-read)
         ;; This is a bit large, but it greatly improved performance.
         (inline bit-reader-read))
(defun bit-reader-read (stream count &optional (eof-error-p t) eof-value)
  "Reads COUNT bits from STREAM, returning an integer.  The integer value is
always interpreted as big endian.  If the end of the stream is reached, and
EOF-ERROR-P is NIL, then an END-OF-FILE condition is raised.  Otherwise
EOF-VALUE is returned."
  (declare (type bit-reader stream)
           (type t/uint32 count)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (let ((ret 0))
    (declare (type (or null #.(if (>= most-positive-fixnum (1- (expt 2 31)))
                                  't/ufixnum
                                  (list 'integer 0 (1- (expt 2 31)))))
                   ret))

    ;; Try to take the easy way out
    (locally
        (declare (optimize speed (debug 0) (safety 0) (compilation-speed 0) (space 0)))
      (when (and (= count 8) (= (%bit-reader-bit-pos stream) 0))
        (when (null (setf ret (%bit-reader-byte stream)))
          ;; We've already reached the end of the stream.
          (if eof-error-p
              (error 'end-of-file :stream (%bit-reader-strm stream))
              (return-from bit-reader-read eof-value)))

        ;; At least one more byte left.
        (setf (%bit-reader-byte stream) (read-byte (%bit-reader-strm stream) nil nil))
        (return-from bit-reader-read ret)))

    (let ((remaining count))
      (declare (type t/uint32 remaining))

      (loop while (plusp remaining)
            for bits-to-read of-type t/ufixnum = (min remaining (- 8 (%bit-reader-bit-pos stream)))
            do (locally
                   (declare (optimize speed (debug 0) (safety 0) (compilation-speed 0) (space 0)))
                 (when (and (plusp remaining)
                            (null (%bit-reader-byte stream)))
                   ;; Nothing left to read
                   (if eof-error-p
                       (error 'end-of-file :stream (%bit-reader-strm stream))
                       (return-from bit-reader-read eof-value))))

               (decf remaining bits-to-read)
               (incf (%bit-reader-bit-pos stream) bits-to-read)
               (logiorf ret (ash (logand (ash (%bit-reader-byte stream)
                                              (- (the t/uint8 (- 8 (%bit-reader-bit-pos stream)))))
                                         (ash #xFF
                                              (- (the t/uint8 (- 8 bits-to-read)))))
                                 remaining))

               (locally
                   (declare (optimize speed (debug 0) (safety 0) (compilation-speed 0) (space 0)))
                 (when (>= (%bit-reader-bit-pos stream) 8)
                   (setf (%bit-reader-bit-pos stream) 0)
                   (setf (%bit-reader-byte stream) (read-byte (%bit-reader-strm stream) nil nil))))))

    ret))

(define-typed-fn bit-reader-advance-to-next-byte ((bit-reader stream))
    (null)
  "Advances to the bit reader to the first bit of the next byte.  If the bit
reader is already on the first bit of the current byte, this does nothing."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (unless (zerop (%bit-reader-bit-pos stream))
    (loop until (zerop (%bit-reader-bit-pos stream)) do
      (unless (bit-reader-read stream 1 nil nil)
        (loop-finish)))
    nil))

(declaim (ftype (function (bit-reader sequence &key (:start fixnum) (:end T)) fixnum)
                bit-reader-read-sequence))
(defun bit-reader-read-sequence (stream sequence &key (start 0) end)
  "Reads bytes into SEQUENCE starting at START up to END, or the end of the
sequence if END is not provided.  The BIT-READER must be on a byte boundary, or
this will raise a restartable NOT-ON-BYTE-ERROR.  This returns the index of the
first element of SEQUENCE that was not updated, which may be less than END if
the end of the stream was reached.

The current BIT-READER-BYTE is always the first byte put into SEQUENCE if it is
non-NIL."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (let ((len 0))
    (declare (type fixnum len)
             (dynamic-extent len))
    (setf len (length sequence))

    (when (or (eq start end) (= (length sequence) 0) (null (%bit-reader-byte stream)))
      ;; Nothing to do
      (return-from bit-reader-read-sequence 0))

    ;; Do a bunch of checks
    (when (< start 0) (error "START cannot be negative"))
    (when (and end (< (the fixnum end) 0)) (error "END cannot be negative"))
    (when (>= start (length sequence)) (error "END is too large for a sequence of size ~a" (length sequence)))
    (when (and end (>= end (length sequence))) (error "END is too large for a sequence of size ~a" (length sequence)))

    ;; Check that we're on a byte boundary
    (restart-case
        (unless (= (%bit-reader-bit-pos stream) 0)
          (error 'not-on-byte-error :format-control "BIT-READER is not on a byte boundary"))
      (advance-to-next-byte ()
        :report "Advance to the next byte"
        (bit-reader-advance-to-next-byte stream)
        (when (null (%bit-reader-byte stream))
          (return-from bit-reader-read-sequence 0))))

    ;; Put the current byte into the destination
    (setf (elt sequence start) (%bit-reader-byte stream))

    (if (or (and end (>= (1+ start) end))
            (>= (1+ start) len))
        ;; Nothing else to do
        (return-from bit-reader-read-sequence 1)

        ;; Read the rest into SEQUENCE
        (prog1 (read-sequence sequence (%bit-reader-strm stream) :start (1+ start) :end (or end len))
          (setf (%bit-reader-byte stream) (read-byte (%bit-reader-strm stream) nil nil))))))

(define-typed-fn bit-reader-read-bytes ((bit-reader stream) (fixnum count) &optional as-list)
    ((or t/uint8-array list))
  "Reads up to COUNT bytes from STREAM into a new sequence and returns it.  If
AS-LIST is truthy, then a LIST is returned, otherwise a (SIMPLE-ARRAY T/UINT8)
is returned."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((ret (make-array 0 :adjustable t :fill-pointer 0 :element-type 't/uint8)))
    (loop for i fixnum from 0 below count
          for byte = (bit-reader-read stream 8 nil nil)
          while byte do (vector-push-extend byte ret)
          finally (return (if as-list
                              (coerce ret 'list)
                              (make-array i :initial-contents ret :element-type 't/uint8))))))

(declaim (ftype (function (bit-reader fixnum &key (:as-list T) (:eof-error-p T) (:eof-value T)) T)
                bit-reader-read-bytes!))
(defun bit-reader-read-bytes! (stream count &key as-list (eof-error-p t) eof-value)
  "Reads COUNT bytes from STREAM into a new sequence and returns it.  If
AS-LIST is truthy, then a LIST is returned, otherwise a (SIMPLE-ARRAY T/UINT8)
is returned.

If this could not read COUNT bytes, and EOF-ERROR-P is truthy, then an
END-OF-FILE condition is raised.  If COUNT bytes could not be read and
EOF-ERROR-P is NIL, then this returns EOF-VALUE."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((ret (make-array 0 :adjustable t :fill-pointer 0 :element-type 't/uint8)))
    (loop for i fixnum from 0 below count
          for byte = (bit-reader-read stream 8 nil nil)
          while byte do (vector-push-extend byte ret)
          finally (return
                    (if (= i count)
                        (if as-list
                            (coerce ret 'list)
                            (make-array i :initial-contents ret :element-type 't/uint8))
                        (if eof-error-p
                            (error 'end-of-file :stream (%bit-reader-strm stream))
                            eof-value))))))

(define-typed-fn bit-reader-read-string ((bit-reader stream) (t/uint32 size) &optional keep-nulls)
    ((values simple-string fixnum) t)
  "Reads up to SIZE bytes, then attempts to convert those bytes into a string
using BYTES->STRING.  On success, this returns two values: the new string, and
the actual number bytes that were read.

KEEP-NULLS is the same as for BYTES->STRING."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((ret (bit-reader-read-bytes stream size)))
    (values (bytes->string ret keep-nulls) (muffling (length ret)))))

(declaim (ftype (function (bit-reader fixnum &key (:keep-nulls T) (:eof-error-p T) (:eof-value T)) T)
                bit-reader-read-string!))
(defun bit-reader-read-string! (stream count &key keep-nulls (eof-error-p t) eof-value)
  "Reads COUNT bytes from STREAM, then attempts to convert those bytes into a
string using BYTES->STRING.  On success, this returns the string.

If this could not read COUNT bytes, and EOF-ERROR-P is truthy, then an
END-OF-FILE condition is raised.  If COUNT bytes could not be read and
EOF-ERROR-P is NIL, then this returns EOF-VALUE.

KEEP-NULLS is the same as for BYTES->STRING."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((seq (bit-reader-read-bytes stream count)))
    (declare (type t/uint8-array seq))
    (if (= (length seq) count)
        (bytes->string seq keep-nulls)
        (if eof-error-p
            (error 'end-of-file :stream (%bit-reader-strm stream))
            eof-value))))

(define-typed-fn bit-reader-peek ((bit-reader stream) (t/uint32 count) &optional eof-error-p eof-value)
    (T)
  "Gets the next COUNT bits from STREAM, but does not read them.  Returns an
integer.  The integer value is always interpreted as big endian.

If the end of the stream is reached, and EOF-ERROR-P is NIL, then an END-OF-FILE
condition is raised.  Otherwise EOF-VALUE is returned."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (let ((old-pos (file-position (%bit-reader-strm stream)))
        (old-bit-pos (%bit-reader-bit-pos stream))
        (old-byte (%bit-reader-byte stream)))
    (unwind-protect
         (bit-reader-read stream count eof-error-p eof-value)
      (file-position (%bit-reader-strm stream) old-pos)
      (setf (%bit-reader-bit-pos stream) old-bit-pos)
      (setf (%bit-reader-byte stream) old-byte))))

(define-typed-fn bit-reader-peek-bytes ((bit-reader stream) (fixnum count) &optional as-list)
    ((or t/uint8-array list))
  "Peeks up to COUNT bytes from STREAM into a new sequence and returns it.  If
AS-LIST is truthy, then a LIST is returned, otherwise a (SIMPLE-ARRAY T/UINT8)
is returned."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (let ((old-pos (file-position (%bit-reader-strm stream)))
        (old-bit-pos (%bit-reader-bit-pos stream))
        (old-byte (%bit-reader-byte stream)))
    (unwind-protect
         (bit-reader-read-bytes stream count as-list)
      (file-position (%bit-reader-strm stream) old-pos)
      (setf (%bit-reader-bit-pos stream) old-bit-pos)
      (setf (%bit-reader-byte stream) old-byte))))

(define-typed-fn bit-reader-count-zeros ((bit-reader stream) &optional discard-first-one)
    (t/ufixnum)
  "Reads bits from the stream until a 1 bit is reached or the end of the stream
is reached, then returns the number of zeros that were read.  If
DISCARD-FIRST-ONE is truthy, then the first 1 bit is also read but not counted,
otherwise it is not read."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (let ((ret 0))
    (declare (type t/ufixnum ret))

    (if discard-first-one
        ;; Slightly faster since we don't do a peek
        (loop while (eq (bit-reader-read stream 1) 0) do (incf ret))

        ;; We need to do a peek if we aren't discarding the first 1 bit.
        (loop for byte = (bit-reader-peek stream 1 nil nil)
              while (and byte (= (the t/uint8 byte) 0)) do
                (bit-reader-read stream 1)
                (incf ret)))
    ret))

(define-typed-fn bit-reader-count-ones ((bit-reader stream) &optional discard-first-zero)
    (t/ufixnum)
  "Reads bits from the stream until a 0 bit is reached or the end of the stream
is reached, then returns the number of ones that were read.  If
DISCARD-FIRST-ZERO is truthy, then the first 0 bit is also read but not counted,
otherwise it is not read."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (let ((ret 0))
    (declare (type t/ufixnum ret))

    (if discard-first-zero
        ;; Slightly faster since we don't do a peek
        (loop while (eq (bit-reader-read stream 1) 1) do (incf ret))

        ;; We need to do a peek if we aren't discarding the first 1 bit.
        (loop for byte = (bit-reader-peek stream 1 nil nil)
              while (and byte (= (the t/uint8 byte) 1)) do
                (bit-reader-read stream 1)
                (incf ret)))
    ret))

(defmacro %define-bit-reader-integer (type size-in-bits signed?)
  (let* ((name (format nil "BIT-READER-READ-~:[U~;~]INT~a" signed? size-in-bits))
         (be-name (format nil "~a/BE" name))
         (read-type (intern (string-upcase (format nil "T/UINT~d" size-in-bits)) :cl-sdm)))
    `(progn
       (define-typed-fn ,(intern name :cl-sdm) ((bit-reader stream))
           (,type t)
         (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
         (nth-value
          0
          (muffling
            ,(if signed?
                 `(uint->int/2c (the ,read-type (bytes->uint (bit-reader-read-bytes! stream ,(/ size-in-bits 8))))
                                ,size-in-bits)
                 `(bytes->uint (bit-reader-read-bytes! stream ,(/ size-in-bits 8)))))))

       (define-typed-fn ,(intern be-name :cl-sdm) ((bit-reader stream))
           (,type t)
         (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
         (nth-value
          0
          (muffling
            ,(if signed?
                 `(uint->int/2c (the ,read-type (bytes->uint (nreverse (bit-reader-read-bytes! stream ,(/ size-in-bits 8)))))
                                ,size-in-bits)
                 `(bytes->uint (nreverse (bit-reader-read-bytes! stream ,(/ size-in-bits 8)))))))))))

(%define-bit-reader-integer t/int16 16 t)
(%define-bit-reader-integer t/uint16 16 nil)

(%define-bit-reader-integer t/int24 24 t)
(%define-bit-reader-integer t/uint24 24 nil)

(%define-bit-reader-integer t/int32 32 t)
(%define-bit-reader-integer t/uint32 32 nil)

(%define-bit-reader-integer t/int64 64 t)
(%define-bit-reader-integer t/uint64 64 nil)
