;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;
;;;; Array Pool
;;;;

(define-condition array-pool-error (simple-error)
  ((array
    :initarg :array
    :reader array-pool-error-array)))

(defclass array-pool ()
  ((type
    :initarg :type
    :initform nil
    :reader array-pool-type)

   (default-value
    :initarg :default-value
    :initform nil
    :accessor array-pool-default-value)

   (arrays
    :initform (make-array 0)
    :type vector)

   (rented
    :initform (make-array 0 :adjustable t :fill-pointer 0 :element-type 'boolean)
    :type (vector boolean)))

  (:documentation "The ARRAY-POOL class provides functionality to allow a
  SIMPLE-ARRAY of a desired size to be 'rented' out so that buffers of the same
  size don't need to be constantly re-created.

  Instances of ARRAY-POOL are not thread safe.  If you need to use array pools
  with multiple threads, create one ARRAY-POOL instance directly per thread."))

(defmethod initialize-instance :after ((pool array-pool) &key &allow-other-keys)
  (setf (slot-value pool 'arrays) (make-array 0 :adjustable t :fill-pointer 0)))

(defun make-array-pool (type default-value)
  "Creates a new ARRAY-POOL instance that will hold SIMPLE-ARRAYs with an
element type of TYPE.  New arrays are created with the default value of
DEFAULT-VALUE.  Existing arrays are not re-initialized between rentings."
  (make-instance 'array-pool :type type :default-value default-value))

(define-typed-fn array-pool-rent ((array-pool pool) (fixnum size))
    (simple-array)
  "Rents a SIMPLE-ARRAY from POOL that has SIZE elements.  The returned array is
not guaranteed to be 'zeroed out' with the default value."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (with-typed-slots ((vector arrays)
                     ((vector boolean) rented)
                     type
                     default-value)
      pool
    (loop with len fixnum = (length arrays)
          for i fixnum from 0 below len
          when (and (not (aref rented i))
                    (= (length (aref arrays i)) size))
            do ;; Found one to return
               (setf (aref rented i) t)
               (return-from array-pool-rent (aref arrays i)))

    ;; None available, make a new array and rent it out.
    (let ((ret (make-array size :element-type type :initial-element default-value)))
      (vector-push-extend ret arrays)
      (vector-push-extend t rented)
      ret)))

(define-typed-fn array-pool-return ((array-pool pool) (simple-array array))
    (null)
  "Lets POOL know that you are done using ARRAY and that it can be marked as not
rented.  After calling this, you should not use ARRAY anymore without re-renting
it first with ARRAY-POOL-RENT.

If ARRAY does not belong to POOL, this will raise a ARRAY-POOL-ERROR condition."
  (declare (optimize (speed 3) (debug 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (with-typed-slots ((vector arrays)
                     ((vector boolean) rented)
                     type
                     default-value)
      pool
    (loop with len fixnum = (length arrays)
          for i fixnum from 0 below len
          when (eq array (aref arrays i)) do
            (setf (aref rented i) nil)
            (return-from array-pool-return))

    (error 'array-pool-error :format-control "Array does not belong to that ARRAY-POOL"
                             :array array)))

(defmacro with-rented-array ((var size pool) &body forms)
  "Rents an array from POOL with SIZE elements and binds it to VAR, then
executes FORMS.  This ensures that the array is returned to POOL at the end.
The final value of FORMS is returned."
  `(let ((,var (array-pool-rent ,pool ,size)))
     (unwind-protect
          (progn ,@forms)
       (array-pool-return ,pool ,var))))
