;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/argparser)

;;;;
;;;; Parsing Functions
;;;;

(define-typed-fn %parse-short-argument ((parser parser) (string arg) (boolean no-error))
    ((or argument null))
  "Parses an individual short argument and returns it, or NIL if it is a positional
argument."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (with-typed-slots ((hash-table args-by-sn))
      parser
    (when (> (length arg) 2)
      ;; Just to make sure, check to see if this is a negative number
      (if (parse-integer? (subseq arg 1))
          (return-from %parse-short-argument) ;; Number
          (argument-error "~a is not a valid short argument" arg))) ;; Not a number

    ;; Try to find the argument by its short name.
    (let ((arg-name (muffling (elt arg 1))))
      (declare (type character arg-name))
      (multiple-value-bind (arg-def found)
          (gethash arg-name args-by-sn)
        (declare (type (or null argument) arg-def))

        ;; Just to make sure, check to see if this is a negative number
        (unless found
          (if (parse-integer? (subseq arg 1))
              ;; Number
              (return-from %parse-short-argument)
              ;; Nope, not a number, and not an argument
              (unless no-error
                (argument-error "~a is not a valid argument" arg))))

        (setf (slot-value arg-def 'called-as) (format nil "-~a" arg-name))
        arg-def))))

(define-typed-fn %parse-long-argument ((parser parser) (string arg))
    ((or argument null) t)
  "Parses an individual long argument and returns it, or NIL if it is a
positional argument."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (with-typed-slots ((hash-table args))
      parser
    (let ((arg-name (subseq arg 2)))
      ;; Try to find the argument
      (multiple-value-bind (arg-def found)
          (gethash arg-name args)
        (declare (type (or null argument) arg-def))
        (when found
          (setf (slot-value arg-def 'called-as) (format nil "--~a" arg-name))
          arg-def)))))

(define-typed-fn %possibly-do-help-or-version ((parser parser) (argument cur-arg) dont-quit)
    (null)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (with-typed-slots ((hash-table args)
                     (string help-long-arg version-long-arg)
                     ((or null character) help-short-arg version-short-arg)
                     (function help-printer version-printer))
      parser
    ;; Was this --help or or -h?
    (when (equalp (arg-name cur-arg) (arg-name (gethash help-long-arg args)))
      (funcall help-printer parser)
      (if dont-quit
          (when (subtypep dont-quit 'error)
            (error dont-quit))
          (exit 0)))

    ;; Was this --version or -V?
    (when (equalp (arg-name cur-arg) (arg-name (gethash version-long-arg args)))
      (funcall version-printer parser)
      (if dont-quit
          (when (subtypep dont-quit 'error)
            (error dont-quit))
          (exit 0))))
  nil)

(defmethod parse-arguments ((parser parser) (argv list)
                            &key dont-quit dont-check-help-or-ver no-positional-arguments &allow-other-keys)
  "Parses ARGV using PARSER.

If DONT-QUIT is non-NIL, then the implicit help and version arguments do not
cause the program to exit.  If DONT-QUIT is a subtype of ERROR, then a condition
of that type will instead be raised.

If DONT-CHECK-HELP-OR-VER is non-NIL, and --help or --version is used, the help
printer/version printer functions are not called.

If NO-POSITIONAL-ARGUMENTS is truthy, then no positional arguments are allowed.
NO-POSITIONAL-ARGUMENTS and ALLOW-DASH-DASH cannot both be truthy at the same
time."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type parser parser)
  (check-type argv list)

  (with-typed-slots ((function help-printer version-printer)
                     (list positional-args)
                     (boolean allow-dash-dash?))
      parser
    (let ((called-args (make-hash-table :test #'equal))
          (cur-arg nil)
          (was-short-p nil)
          (state :read-arg))
      (declare (type (member :read-arg :read-token :read-positional) state)
               (type hash-table called-args)
               (type (or null argument) cur-arg)
               (type boolean was-short-p))

      ;; Start parsing
      (dolist (arg (rest argv))
        (declare (type simple-string arg))

        (tagbody
           (ecase state
             (:read-arg
              ;; Determine if we need to parse an argument
              (cond
                ((char= (elt arg 0) #\-)
                 (setf was-short-p nil)

                 ;; See if this is a long or short argument
                 (cond
                   ((= (length arg) 1)
                    ;; Just a single "-", add to positional arguments
                    (if no-positional-arguments
                        (argument-error "Unknown argument: ~a" arg)
                        ;; It looked like a short argument but wasn't handled yet,
                        ;; so it's a positional argument that happens to be a
                        ;; negative number.
                        (setf positional-args (nconc positional-args (list arg))))
                    (go ==next==))

                   ((not (char= (elt arg 1) #\-))
                    (progn
                      (setf cur-arg (%parse-short-argument parser arg nil))
                      (setf was-short-p t)))

                   (t
                    (setf cur-arg (%parse-long-argument parser arg))))

                 ;; Check to see if this was an actual argument, or a positional
                 ;; argument that is a negative number, or the special '--' argument.
                 (cond
                   (cur-arg
                    (setf (slot-value cur-arg 'called?) t)
                    (case (arg-type cur-arg)
                      ;; Flags have no token, so we can set those here
                      (:flag
                       ;; Flags can only be used once
                       (cond
                         ((gethash (arg-name cur-arg) called-args)
                          (argument-error "~a cannot be used more than once" (slot-value cur-arg 'called-as)))
                         (t
                          (setf (gethash (arg-name cur-arg) called-args) cur-arg)))

                       (setf (slot-value cur-arg 'value) t)
                       (unless dont-check-help-or-ver
                         (%possibly-do-help-or-version parser cur-arg dont-quit)))

                      ;; Multi-Flags also have no token, so we can set
                      ;; those here, as well.
                      (:multi-flag
                       (if (not (arg-value cur-arg))
                           (setf (slot-value cur-arg 'value) 1)
                           (incf (the (and unsigned-byte fixnum) (slot-value cur-arg 'value)))))

                      ;; Anything else needs a token
                      (otherwise (setf state :read-token))))

                   (was-short-p
                    (if no-positional-arguments
                        (argument-error "Unknown argument: ~a" arg)
                        ;; It looked like a short argument but wasn't handled yet,
                        ;; so it's a positional argument that happens to be a
                        ;; negative number.
                        (setf positional-args (nconc positional-args (list arg)))))

                   ((and allow-dash-dash? (muffling (string= arg "--")))
                    ;; Start of positional arguments
                    (setf state :read-positional))

                   (t
                    ;; Invalid argument
                    (argument-error "Unknown argument: ~a" arg))))

                ;; Anything not starting with a dash is regarded as a
                ;; positional argument
                (t
                 (if no-positional-arguments
                     (argument-error "Unknown argument: ~a" arg)
                     (setf positional-args (nconc positional-args (list arg))))))) ;; End of :READ-ARG state

             (:read-token
              ;; Check to see if we got another valid argument where we
              ;; expected a token.  We purposely try to fail here.  The
              ;; exception is if there is only a dash on its own, which
              ;; is ignored since it's therefore the argument value.
              (if (and (> (length arg) 1)
                       (char= (elt arg 0) #\-))

                  (if (not (char= (elt arg 1) #\-))
                      (when (%parse-short-argument parser arg t)
                        (argument-error "--~a expects a value" (arg-name cur-arg)))

                      (when (%parse-long-argument parser arg)
                        (argument-error "--~a expects a value" (arg-name cur-arg)))))

              (ecase (arg-type cur-arg)
                (:string
                 (when (gethash (arg-name cur-arg) called-args)
                   (argument-error "~a cannot be used more than once" (slot-value cur-arg 'called-as)))
                 (setf (gethash (arg-name cur-arg) called-args) cur-arg)
                 (setf (slot-value cur-arg 'called?) t)
                 (unless (check-arg-value-against-constraint cur-arg arg)
                   (error 'argument-error
                          :format-control (format nil "~a can only accept one of the following: ~{~a~^ ~}"
                                                  (slot-value cur-arg 'called-as) (arg-constraint cur-arg))))
                 (setf (slot-value cur-arg 'value) arg))

                (:multi-string
                 (unless (check-arg-value-against-constraint cur-arg arg)
                   (argument-error "~a can only accept arguments of the following: ~{~a~^ ~}"
                                   (slot-value cur-arg 'called-as) (arg-constraint cur-arg)))
                 (setf (slot-value cur-arg 'value) (nconc (slot-value cur-arg 'value) (list arg)))
                 (setf (gethash (arg-name cur-arg) called-args) cur-arg)
                 (setf (slot-value cur-arg 'called?) t))

                (:integer
                 (when (gethash (arg-name cur-arg) called-args)
                   (argument-error "~a cannot be used more than once" (slot-value cur-arg 'called-as)))
                 (setf (gethash (arg-name cur-arg) called-args) cur-arg)
                 (setf (slot-value cur-arg 'called?) t)

                 (handler-case
                     (setf (slot-value cur-arg 'value) (parse-integer arg))
                   (parse-error ()
                     (argument-error "~a expects an integer" (slot-value cur-arg 'called-as))))

                 ;; Use ARG-VALUE here because it now stores the parsed integer.
                 (unless (check-arg-value-against-constraint cur-arg (slot-value cur-arg 'value))
                   (argument-error "~a expects an integer between ~d and ~d"
                                   (slot-value cur-arg 'called-as)
                                   (car (arg-constraint cur-arg))
                                   (cdr (arg-constraint cur-arg)))))

                (:float
                 (when (gethash (arg-name cur-arg) called-args)
                   (argument-error "~a cannot be used more than once" (slot-value cur-arg 'called-as)))
                 (setf (gethash (arg-name cur-arg) called-args) cur-arg)
                 (setf (slot-value cur-arg 'called?) t)
                 (handler-case
                     (setf (slot-value cur-arg 'value) (parse-float arg :type 'double-float))
                   (parse-error ()
                     (argument-error "~a expects a float" (slot-value cur-arg 'called-as))))

                 ;; Use ARG-VALUE here because it now stores the parsed integer.
                 (unless (check-arg-value-against-constraint cur-arg (slot-value cur-arg 'value))
                   (argument-error "~a expects a float between ~a and ~a"
                                   (slot-value cur-arg 'called-as)
                                   (car (arg-constraint cur-arg))
                                   (cdr (arg-constraint cur-arg)))))


                ((:file :existing-file)
                 ;; We check for file existence as-needed later
                 (when (gethash (arg-name cur-arg) called-args)
                   (argument-error "~a cannot be used more than once" (slot-value cur-arg 'called-as)))
                 (setf (gethash (arg-name cur-arg) called-args) cur-arg)
                 (setf (slot-value cur-arg 'called?) t)
                 (setf (slot-value cur-arg 'value) (uiop:parse-native-namestring arg)))

                ((:multi-file :multi-existing-file)
                 ;; We check for file existence as-needed later
                 (setf (slot-value cur-arg 'value) (nconc (slot-value cur-arg 'value) (list (uiop:parse-native-namestring arg))))
                 (setf (gethash (arg-name cur-arg) called-args) cur-arg)
                 (setf (slot-value cur-arg 'called?) t)))

              (setf cur-arg nil)
              (setf state :read-arg))

             (:read-positional
              (if no-positional-arguments
                  (argument-error "Unknown argument: ~a" arg)
                  (setf positional-args (nconc positional-args (list arg))))))
           ==next==))

      ;; We left the loop too early :^(
      (when (eq state :read-token)
        (argument-error "--~a expects a value" (arg-name cur-arg)))

      ;; Simple way to keep track if any arguments were called
      (setf (slot-value parser 'arguments-used?)
            (> (hash-table-count called-args) 0))

      ;; Check args (check functions, required args, etc)
      (maphash #'(lambda (name arg)
                   (declare (ignore name)
                            (type argument arg))
                   (cond
                     ((slot-value arg 'called?)
                      (case (arg-type arg)
                        (:existing-file
                         ;; Check that the file exists
                         (unless (probe-file (arg-value arg))
                           (argument-error "File does not exist: ~a" (arg-value arg))))

                        (:multi-existing-file
                         ;; Check that all files passed in exist
                         (dolist (file (arg-value arg))
                           (unless (probe-file file)
                             (argument-error "File does not exist: ~a" file))))

                        (otherwise
                         (when-var (fn (arg-checking-fn arg))
                           ;; Run the argument checking function
                           (funcall (the function fn) arg)))))

                     (t ;; Not called, see if we have a required argument string for it
                      (when-var (str (arg-not-found-str arg))
                        (argument-error "~a" str)))))
               (the hash-table (slot-value parser 'args)))))

  ;; All done
  nil)

(defmacro with-args ((args-var &rest make-parser-args)
                     (&key fake-argv dont-quit?)
                     (&rest arg-defs)
                     (on-error-fn)
                     &body forms)
  "Binds ARGS-VAR to a new PARSER, constructing the new PARSER using
MAKE-PARSER-ARGS.  ARG-DEFS are then expanded tino calls to DEFARGUMENT.  Then,
PARSE-ARGUMENTS is called with ARGS-VAR.

If FAKE-ARGV is supplied, the call to PARSE-ARGUMENTS will use FAKE-ARGV as the
arguments to parse.  Thus it should contain a fake program name in index 0.  If
FAKE-ARGV is not supplied, it will use the implementation command line arguments
as usual.

DONT-QUIT is the same as in PARSE-ARGUMENTS.

If any ARGUMENT-ERROR conditions are raised during argument parsing, ON-ERROR-FN
will be called with the ARGUMENT-ERROR instance as the sole parameter.

Finally, FORMS is executed after arguments are parsed."

  `(let ((,args-var (make-parser ,@make-parser-args)))
     ,@(loop for arg-def in arg-defs
             collect (append (list 'defargument args-var)
                             arg-def))

     (handler-bind
         ((argument-error ,on-error-fn))
       (parse-arguments ,args-var (if (listp ,fake-argv)
                                      (if (> (length ,fake-argv) 0)
                                          ,fake-argv
                                          (list "fake-args-used"))
                                      (get-implementation-args))
                        :dont-quit ,dont-quit?)
       ,@forms)))
