;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/argparser)

;;;;
;;;; Parsing Classes
;;;;

;;;
;;; BASE-PARSER Class
;;;

(defclass base-parser ()
  ((allow-dash-dash?
    :initarg :allow-dash-dash
    :initform nil
    :type boolean
    :reader parser-allow-dash-dash-p
    :documentation "During parsing, when this is T, then two hyphens (\"--\") can be used to
indicate the end of all arguments, and that all subsequent arguments should be
considered positional arguments.  Otherwise, \"--\" will be considered an
invalid argument.")

   (arguments-used?
    :initform nil
    :type boolean
    :reader parser-arguments-used-p
    :documentation "Returns T if any arguments were used after parsing is complete, or NIL
otherwise.  This is always NIL if the arguments have not yet been parsed.")

   (positional-args
    :initform ()
    :type list
    :reader parser-positional-args
    :documentation "The list of positional arguments encountered during parsing.  This is always an
empty list if the arguments haven't yet been parsed.")

   (args
    :initform (make-hash-table :test #'equalp :size 3)
    :type hash-table
    :documentation "The internal table of defined ARGUMENTs, referenced by their long name (minus the '--').")

   (args-by-sn
    :initform (make-hash-table :test #'equal :size 3)
    :type hash-table
    :documentation "The internal table of defined ARGUMENTs, referenced by their short name (minus the '-')."))

  (:documentation "The base class for argument parsers.  This doesn't provide anything fancy, and
is just basic 'I have arguments that I want parsed' functionality that other
classes can build upon.

If you are building a command line program, then you want the PARSER class instead."))

(defmethod initialize-instance :after ((parser base-parser) &key &allow-other-keys)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-slots (allow-dash-dash?) parser
    ;; Normalize to BOOLEAN
    (setf allow-dash-dash? (if allow-dash-dash? t nil))))

(defmethod print-object ((obj base-parser) out)
  (print-unreadable-object (obj out :type t)
    (format out "Num arguments: ~:d" (hash-table-count (slot-value obj 'args)))))

;;;
;;; PARSER Class
;;;

(defclass parser (base-parser)
  ((program-name
    :initarg :program-name
    :initform ""
    :type string
    :reader parser-program-name
    :documentation "The simple name of the program (usually this is the binary name for command line
programs).  This is mainly used by the default help printing function.")

   (program-fancy-name
    :initarg :program-fancy-name
    :initform ""
    :type string
    :reader parser-program-fancy-name
    :documentation "The fancy name of the program.  This is mainly used by the default version
printing function.")

   (program-version
    :initarg :program-version
    :initform "v???"
    :type string
    :reader parser-program-version
    :documentation "The version of the program.  This is mainly used by the default version printing
function.")

   (help-pre-text
    :initarg :help-pre-text
    :initform nil
    :type (or null string)
    :reader parser-help-pre-text
    :documentation "Used by the default help printer.  This is text that will be printed after the
usage line (followed by a newline), but before the list of all possible
arguments, if this is non-NIL and not an empty string.  The exact output format
used by the default printer is:

<usage line>

<HELP-PRE-TEXT>
Available Options:
<list of all options>
<HELP-POST-TEXT>")

   (help-post-text
    :initarg :help-post-text
    :initform nil
    :type (or null string)
    :reader parser-help-post-text
    :documentation "Used by the default help printer.  This is text that will be printed after the
list of all possible arguments if this is non-NIL and not an empty string.  The
exact output format used by the default printer is:

<usage line>

<HELP-PRE-TEXT>
Available Options:
<list of all options>
<HELP-POST-TEXT>")

   (version-post-text
    :initarg :version-post-text
    :initform nil
    :type (or null string)
    :reader parser-version-post-text
    :documentation "Used by the default version printer.  This is text that will be printed after
the version information is printed when this is non-NIL and not an empty string.
The exact output format used by the default printer is:

<program name> <program version>
<VERSION-POST-TEXT>")

   (usage-string
    :initarg :usage-string
    :initform nil
    :type (or null string)
    :reader parser-usage-string
    :documentation "When this is non-NIL, then this will be used by the default help printer as an
alternate 'Usage:' line instead of the default built-in text.")

   (help-short-arg
    :initarg :help-short-arg
    :initform +default-help-short-name+
    :type (or null character)
    :reader parser-help-short-arg
    :documentation "The short name of the built-in '--help' argument.  This cannot be '-', a digit,
or a non-printable character.  When this is NIL, then the --help argument has no
short equivalent.")

   (help-long-arg
    :initarg :help-long-arg
    :initform +default-help-long-name+
    :type string
    :reader parser-help-long-arg
    :documentation "The long name of the built-in '--help' argument.  This cannot be empty once the
hyphens are removed.")

   (version-short-arg
    :initarg :version-short-arg
    :initform +default-version-short-name+
    :type (or null character)
    :reader parser-version-short-arg
    :documentation "The short name of the built-in '--version' argument.  This cannot be '-', a
digit, or a non-printable character.  When this is NIL, then the --version
argument has no short equivalent.")

   (version-long-arg
    :initarg :version-long-arg
    :initform +default-version-long-name+
    :type string
    :reader parser-version-long-arg
    :documentation "The long name of the built-in '--version' argument.  This cannot be empty once
the hyphens are removed.")

   (help-printer
    :initarg :help-printer
    :initform #'default-help-printer
    :type function
    :reader parser-help-printer
    :documentation "The function that will be called when the built-in '--help' command line
argument is encountered.  This must accept one argument, the PARSER instance.")

   (version-printer
    :initarg :version-printer
    :initform #'default-version-printer
    :type function
    :reader parser-version-printer
    :documentation "The function that will be called when the built-in '--version' command line
argument is encountered.  This must accept one argument, the PARSER instance."))
  (:documentation "A class used to parse command line arguments.  You can use either MAKE-INSTANCE
or MAKE-PARSER to create an instance of this class."))

(defmethod print-object ((obj parser) out)
  (print-unreadable-object (obj out :type t)
    (format out "for \"~a ~a\", Num Arguments: ~:d"
            (parser-program-name obj)
            (parser-program-version obj)
            (hash-table-count (slot-value obj 'args)))))

(defmethod initialize-instance :after ((parser parser) &key &allow-other-keys)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (with-slots (program-name program-fancy-name program-version help-pre-text help-post-text version-post-text
               usage-string help-short-arg help-long-arg version-short-arg version-long-arg help-printer
               version-printer)
      parser
    (check-type program-name string)
    (check-type program-fancy-name string)
    (check-type program-version string)
    (check-type help-pre-text (or null string))
    (check-type help-post-text (or null string))
    (check-type version-post-text (or null string))
    (check-type usage-string (or null string))
    (check-type help-short-arg (or null character))
    (check-type version-short-arg (or null character))
    (check-type help-long-arg string)
    (check-type version-long-arg string)
    (check-type help-printer function)
    (check-type version-printer function)

    ;; Check short names
    (when help-short-arg
      (cond
        ((char= help-short-arg #\-)
         (error "The HELP-SHORT-NAME cannot be #\-"))

        ((not (alpha-char-p help-short-arg))
         (error "Invalid HELP-SHORT-ARG"))))

    (when version-short-arg
      (cond
        ((char= version-short-arg #\-)
         (error "The VERSION-SHORT-NAME cannot be #\-"))

        ((not (alpha-char-p version-short-arg))
         (error "Invalid VERSION-SHORT-ARG"))))

    ;; Adjust help and version long arguments
    (when (string-starts-with help-long-arg "--")
      (setf help-long-arg (subseq (the string help-long-arg) 2)))

    (when (string-starts-with version-long-arg "--")
      (setf version-long-arg (subseq (the string version-long-arg) 2)))

    ;; Check long names
    (cond
      ((empty-string-p! help-long-arg)
       (error "HELP-LONG-NAME cannot be empty"))
      ((empty-string-p! version-long-arg)
       (error "VERSION-LONG-NAME cannot be empty")))

    ;; Define the --help and --version arguments
    (defargument parser help-long-arg
      :short-name help-short-arg
      :type :flag
      :help "This help text")

    (defargument parser version-long-arg
      :short-name version-short-arg
      :type :flag
      :help
      "Displays version information")))

(defun make-parser (&key help-printer version-printer
                      program-name program-fancy-name program-version
                      (help-pre-text "") (help-post-text "")
                      (version-post-text "")
                      (help-short-arg +default-help-short-name+)
                      (help-long-arg +default-help-long-name+)
                      (version-short-arg +default-version-short-name+)
                      (version-long-arg +default-version-long-name+)
                      usage-string
                      allow-dash-dash)
  "Creates a new PARSER instance."
  (declare (type (or null function) help-printer version-printer)
           (type (or null string) program-name program-fancy-name program-version)
           (type string help-pre-text help-post-text version-post-text help-long-arg version-long-arg)
           (type (or null character) help-short-arg version-short-arg)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (let ((prognam (or program-name
                     (file-namestring (first
                                       #+sbcl sb-ext:*posix-argv*
                                       #+clisp ext:*args*
                                       #+ccl ccl:*command-line-argument-list*
                                       #+ecl sys:*command-args*)))))
    (make-instance
     'parser
     :program-name       prognam
     :program-fancy-name (or program-fancy-name "")
     :program-version    (or program-version "v???")
     :help-pre-text      help-pre-text
     :help-post-text     help-post-text
     :version-post-text  version-post-text
     :help-printer       (or help-printer #'default-help-printer)
     :version-printer    (or version-printer #'default-version-printer)
     :help-short-arg     help-short-arg
     :help-long-arg      help-long-arg
     :version-short-arg  version-short-arg
     :version-long-arg   version-long-arg
     :usage-string       usage-string
     :allow-dash-dash    allow-dash-dash)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; PARSER-related Functions
;;;

(defun defargument (parser long-name &key (type :string) short-name
                                       (help "")
                                       (group "")
                                       (init-val nil init-val-supplied-p)
                                       check-fn
                                       not-found-str
                                       constraint)
  "Creates a new ARGUMENT of argument type TYPE and adds it to PARSER.  This
returns the new argument instance.  Providing NOT-FOUND-STR creates a required
argument (see below).

LONG-NAME the new long name of the argument.  This should not include any
leading hyphens, and must be a string.

When SHORT-NAME is provided, it must be a character.  This is the short name for
the argument.  If it is not provided or is NIL, the argument will have no short
name.

HELP is the help message associated with the argument.  It must be a string.

GROUP is used to group arguments together.  When using the default help printer,
arguments will be grouped by GROUP.  This must be a string.

When INIT-VAL is provided, it becomes the default value of the argument.

When CHECK-FN is provided, it should be a function that should take an ARGUMENT
parameter.  This is called by PARSE-ARGS after parsing is complete, and is meant
to be used as a way to customize argument value checks.  Having CHECK-FN raise
an ARGUMENT-ERROR is the recommended way to indicate that CHECK-FN has failed.

When NOT-FOUND-STR is provided, it must be a string.  This is used as an error
message in the case that this argument was not called.  Thus when this is
provided, the argument becomes a required argument.

When CONSTRAINT is provided, its behavior depends on the TYPE:

* :FLAG - CONSTRAINT is always ignored.

* :MULTI-FLAG - CONSTRAINT is always ignored.

* :STRING - CONSTRAINT should be a list, and the argument's value after parsing
  must be one of the given values.  Always case sensitive.

* :MULTI-STRING - CONSTRAINT should be a list, and each argument value after parsing
  must be one of the given values.  Always case sensitive.

* :FILE - CONSTRAINT is always ignored.

* :EXISTING-FILE - CONSTRAINT is always ignored.

* :MULTI-FILE - CONSTRAINT is always ignored.

* :MULTI-EXISTING-FILE - CONSTRAINT is always ignored.

* :INTEGER - CONSTRAINT should be a CONS, where the CAR is the minimum allowed
  value, and the CDR is the maximum allowed value, inclusive.

* :FLOAT - CONSTRAINT should be a CONS, where the CAR is the minimum allowed
  value, and the CDR is the maximum allowed value, inclusive."
  (declare (type string long-name help group)
           (type (or character null) short-name)
           (type t/argument-type type)
           (type (or null function) check-fn)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (let ((new-arg
          ;; We need to check INIT-VAL-SUPPLIED-P here since we later check if
          ;; the slot is bound or not in the INITIALIZE-INSTANCE :AFTER method.
          (if init-val-supplied-p
              (make-instance 'argument
                             :name (trim-whitespace long-name)
                             :short-name short-name
                             :type type
                             :help help
                             :group group
                             :constraint constraint
                             :init-value init-val
                             :checking-fn check-fn
                             :not-found-str not-found-str)

              ;; No initial value
              (make-instance 'argument
                             :name (trim-whitespace long-name)
                             :short-name short-name
                             :type type
                             :help help
                             :group group
                             :constraint constraint
                             :checking-fn check-fn
                             :not-found-str not-found-str))))

    (with-typed-slots ((hash-table args args-by-sn))
        parser
      (multiple-value-bind (val foundp)
          (gethash (arg-name new-arg) args nil)
        (declare (ignore val))
        (when foundp
          (argdef-error "Duplicate argument definitions: ~a" (arg-name new-arg))))

      (when-var (short-name (arg-short-name new-arg))
        (multiple-value-bind (val foundp)
            (gethash short-name args-by-sn nil)
          (declare (ignore val))
          (when foundp
            (argdef-error "Duplicate argument definitions: ~a" short-name))))

      (when short-name
        (setf (gethash short-name args-by-sn) new-arg))
      (setf (gethash (arg-name new-arg) args) new-arg))
    new-arg))

(define-typed-fn get-arg-value ((parser parser) (string long-name))
    (t t)
  "Returns the current value of the argument named LONG-NAME in PARSER.  If
LONG-NAME is not the name of a valid argument, then an ARGDEF-ERROR is raised.
LONG-NAME should not include any leading hyphens.

If the argument was not called, this returns either the default value or NIL if
the argument has no default.

If the argument was called, the value returned depends on the type of argument:

* :FLAG - Returns T if the argument was called, or NIL otherwise.

* :MULTI-FLAG - Returns an integer indicating how many times the argument was
  called.

* :STRING - Returns a string.

* :MULTI-STRING - Returns a list of strings.

* :INTEGER - Returns an INTEGER.

* :FLOAT - Returns a DOUBLE-FLOAT.

* :FILE - Returns a PATHNAME

* :EXISTING-FILE - Returns a PATHNAME.

* :MULTI-FILE - Returns a list of PATHNAMEs.

* :MULTI-EXISTING-FILE - Returns a list of PATHNAMEs."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (with-typed-slots ((hash-table args))
      parser
    (multiple-value-bind (arg found)
        (gethash long-name args)
      (unless found
        (argdef-error "Invalid argument: ~a~%" long-name))
      (arg-value arg))))

(define-typed-fn arg-called-p ((parser parser) (string long-name))
    (boolean t)
  "Returns T if the argument named LONG-NAME was called during parsing, or NIL
otherwise.  If LONG-NAME is not the name of a valid argument, then an
ARGDEF-ERROR is raised.  LONG-NAME should not include any leading hyphens."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((hash-table args))
      parser
    (multiple-value-bind (arg found)
      (gethash long-name args)
      (unless found
        (argdef-error "Invalid argument: ~a~%" long-name))
      (slot-value arg 'called?))))

(defun get-integer-arg (parser long-name &key extra-test error-text (radix 10))
  "Attempts to parse an integer from the given argument's value.

EXTRA-TEST is an optional additional test function that will be called upon a
successful parse (for example, PLUSP to check for a positive integer).  The
function should take one argument.  If EXTRA-TEST fails, an ARGUMENT-ERROR with
the text 'Invalid value passed to [long name]' will be raised.

If an integer cannot be parsed, an ARGUMENT-ERROR will be raised with text
stating that the argument expects an integer.  If ERROR-TEXT is provided, that
format string will be used instead.  The format string must allow for a single
string argument."
  (declare (type (or function null) extra-test)
           (type fixnum radix)
           (type parser parser)
           (type string long-name)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (handler-case
      (let ((ret (parse-integer (get-arg-value parser long-name) :radix radix)))
        (when extra-test
          (unless (funcall extra-test ret)
            (argument-error "Invalid value passed to --~a" long-name)))
        ret)
    (parse-error ()
      (argument-error (if error-text
                          error-text
                          "--~a expects an integer")
                      long-name))))

(defmacro with-called-arg ((parser arg-name value-var) &body forms)
  "Binds the value of the argument named ARG-NAME to VALUE-VAR, then executes
FORMS."
  `(when (arg-called-p ,parser ,arg-name)
     (let ((,value-var (get-arg-value ,parser ,arg-name)))
       (declare (ignorable ,value-var))
       ,@forms)))

(defmacro let-args (parser bindings &body forms)
  "Binds multiple arguments to values, then executes FORMS.

BINDINGS should be a list of binding definitions, where each definition is a
list in the form (VAR-NAME ARG-NAME &OPTIONAL DEFAULT-VALUE).  If DEFAULT-VALUE
is supplied, then VAR-NAME gets the value of DEFAULT-VALUE only if the value of
the arg named ARG-NAME is NIL."
  `(let (,@(loop for (var-name arg-name default) in bindings
                 if default
                   collect `(,var-name (or (get-arg-value ,parser ,arg-name)) ,default)
                 else
                   collect `(,var-name (get-arg-value ,parser ,arg-name))))
     ,@forms))

(defmacro let-args* (parser bindings &body forms)
    "Binds multiple arguments to values in the order defined in BINDINGS, then
executes FORMS.

BINDINGS should be a list of binding definitions, where each definition is a
list in the form (VAR-NAME ARG-NAME &OPTIONAL DEFAULT-VALUE).  If DEFAULT-VALUE
is supplied, then VAR-NAME gets the value of DEFAULT-VALUE only if the value of
the arg named ARG-NAME is NIL."
  `(let* (,@(loop for (var-name arg-name default) in bindings
                  if default
                    collect `(,var-name (or (get-arg-value ,parser ,arg-name)) ,default)
                  else
                    collect `(,var-name (get-arg-value ,parser ,arg-name))))
     ,@forms))

(defmacro if-arg ((parser arg-name value-var) have-arg-form &optional not-have-arg-form)
  "Checks to see if ARG-NAME was called in PARSER.  If it was, VALUE-VAR is locally
bound to that argument's value and HAVE-ARG-FORM is executed.  Otherwise,
NOT-HAVE-ARG-FORM is executed instead and VALUE-VAR is not bound."
  `(if (arg-called-p ,parser ,arg-name)
       (let ((,value-var (get-arg-value ,parser ,arg-name)))
         (declare (ignorable ,value-var))
         ,have-arg-form)
       ,not-have-arg-form))

(trivial-indent:define-indentation if-arg (4 2 2))

(define-typed-fn parser-set-arg-value ((parser parser) (string name) value)
    (t t)
  "Sets the value of the argument named NAME in ARGS.  NAME must be the long name
of the argumenmt.  This function does not check that VALUE matches the type of
the argument.

Note: This allows you to set an argument to a value that may violate its
constraints."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((hash-table args))
      parser
    (multiple-value-bind (arg found)
        (gethash name args)
      (unless found
        (argdef-error "Invalid argument: ~a~%" name))
      (setf (slot-value arg 'value) value))))

(define-typed-fn arg-called-as ((parser parser) (string arg-name))
    ((or null string) t)
  "If the argument named ARG-NAME was called, this returns a string containing
how it was called.  Otherwise this returns NIL.  ARG-NAME must be the long name
of the argument."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (if (arg-called-p parser arg-name)
      (the string
           (slot-value (the argument (gethash arg-name (the hash-table (slot-value parser 'args))))
                       'called-as))
      nil))

(define-typed-fn parser-reset ((parser parser))
    (null)
  "Resets PARSER to a clean state.  All arguments are set to their initial
values, and positional arguments are cleared."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((hash-table args)
                     (list positional-args)
                     (boolean arguments-used?))
      parser
    (maphash #'(lambda (name arg)
                 (declare (ignore name)
                          (type argument arg))
                 (setf (slot-value arg 'value) (if (slot-boundp arg 'init-value)
                                                   (arg-init-value arg)
                                                   nil))
                 (setf (slot-value arg 'called?) nil)
                 (setf (slot-value arg 'called-as) ""))
             args)
    (setf positional-args nil)
    (setf arguments-used? nil))
  nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Help and Version Printing
;;;

(define-typed-fn write-indent ((fixnum len) (stream stream))
    (null t)
  (declare (optimize speed (safety 0) (debug 1) (compilation-speed 0) (space 0)))
  (dotimes (i len) (write-char #\Space stream))
  nil)

(define-typed-fn write-argument-description ((string str) (fixnum offset) (string-stream stream))
    (null t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (let* ((line-limit 0)
         (orig-offset 0)
         (first-line t)
         (real-str str))
    (declare (type fixnum line-limit orig-offset)
             (type simple-string real-str)
             (type boolean first-line)
             (dynamic-extent line-limit orig-offset first-line))
    (setf line-limit (- 79 offset))
    (setf orig-offset line-limit)

    ;; If the description can fit all on this line, just write it out and return
    (when (>= line-limit (length real-str))
      (format stream "~a~%" real-str)
      (return-from write-argument-description))

    (loop do
      (if (< line-limit (length real-str))
          (if (or (equal (elt real-str line-limit) #\Space)
                  (equal (elt real-str line-limit) #\Newline))
              (progn
                (when (not first-line)
                  (write-indent (if (= offset 0) offset (1+ offset)) stream))
                (format stream "~a~%" (trim-whitespace (subseq real-str 0 line-limit)))
                (setf real-str (subseq real-str line-limit))
                (setf line-limit orig-offset)
                (setf first-line nil))

              (decf line-limit))

          (progn
            (write-indent (if (= offset 0) offset (1+ offset)) stream)
            (format stream "~a~%" (trim-whitespace real-str))
            (setf first-line nil)
            (loop-finish)))))
  nil)

(defun parser-get-argument-groups (parser)
  "Returns a (VECTOR STRING) of all the argument groups contained in PARSER's ARGUMENTs."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type parser parser)

  (with-typed-slots ((hash-table args))
      parser
    (let ((ret (make-hash-table :test #'equalp)))
      (maphash #'(lambda (arg-name arg)
                   (declare (type string arg-name)
                            (type argument arg))
                   (when (not (gethash (arg-group arg) ret))
                     (setf (gethash (arg-group arg) ret) (new-vector string)))

                   (vector-push-extend arg-name
                                       (the (vector string)
                                            (gethash (the string (arg-group arg)) ret))))
               args)
      ret)))

(defmethod default-help-printer ((parser parser))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (with-typed-slots ((hash-table args)
                     (string program-name)
                     ((or null string) usage-string help-pre-text help-post-text))
      parser
    (let ((groups (parser-get-argument-groups parser))
          (longest-long-arg 0)
          (arg nil)
          (arg-name ""))
      (declare (type (and unsigned-byte fixnum) longest-long-arg)
               (type string arg-name)
               (type (or null argument) arg))

      ;; Start the output
      (with-output-to-string (out)
        ;; Pring the usage line.
        (if usage-string
            (format out "~a~%~%" usage-string)
            (format out "Usage: ~a [options]~%~%" program-name))

        ;; Print pre-help text
        (when (and help-pre-text
                   (not (empty-string-p help-pre-text)))
          (format out "~a~%" help-pre-text))

        (format out "Available Options:~%")

        ;; Find the length of the longest argument long name
        (loop for arg being the hash-keys in args
              do (when (> (length (the string arg)) longest-long-arg)
                   (setf longest-long-arg (length (the string arg)))))

        (loop for cat string being the hash-keys in groups do
          (format out "~%~a~%" (if (string= cat "") "General Options" cat))
          (format out "================================================================================~%")

          (let ((group-seq (gethash cat groups)))
            (declare (type (vector string) group-seq))

            (dotimes (i (length group-seq))
              (muffling (setf arg-name (aref group-seq i)))
              (setf arg (gethash arg-name args))

              ;; Write the long name
              (format out "--~a" (the string (arg-name arg)))

              ;; Write an "x" if it's not a flag
              (case (arg-type arg)
                ((:flag :multi-flag)
                 (write-string "  " out)
                 (write-indent (- longest-long-arg (length (the string (arg-name arg))))
                               out))

                (otherwise
                 (write-string " x" out)
                 (write-indent (- longest-long-arg (length (the string (arg-name arg))))
                               out)))

              (with-typed-slots (((or character null) short-name)
                                 (string help))
                  arg
                ;; Write the short name, if there is one
                (cond
                  (short-name
                   ;; Write the short name.
                   (write-string " / -" out)
                   (write-char short-name out)

                   ;; Write the extra "x' when it's not a flag
                   (case (arg-type arg)
                     ((:flag :multi-flag)
                      (write-string "   : " out))
                     (otherwise
                      (write-string " x : " out))))

                  (t ;; No short name, just write the divider
                   (write-indent 5 out)
                   (write-string "   : " out)))

                ;; Write the argument's description
                (write-argument-description help (+ longest-long-arg 13) out)))))

        ;; Print post-help text
        (when (and help-post-text (not (empty-string-p help-post-text)))
          (format out "~%~a~%" (parser-help-post-text parser)))

        (format *standard-output* "~a" (get-output-stream-string out)))))
  nil)

(defmethod default-version-printer ((parser parser))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((string program-name program-fancy-name program-version)
                     ((or null string) version-post-text))
      parser
    (format *standard-output* "~a ~a~%"
            (if (empty-string-p! program-fancy-name)
                program-name
                program-fancy-name)
            program-version)

    (when (and version-post-text (not (empty-string-p version-post-text)))
      (format *standard-output* "~a~%" version-post-text)))
  nil)
