;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/argparser)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Types and Constants
;;;

(defining-consts
  (+default-help-short-name+ #\h :documentation "The default short name for the --help argument.")
  (+default-help-long-name+ "help" :documentation "The default long name for the --help argument.")
  (+default-version-short-name+ #\V :documentation "The default short name for the --version argument.")
  (+default-version-long-name+ "version" :documentation "The default long name for the --version argument."))

(deftype t/argument-type ()
  "The various types of supported arguments."
  '(member
    :flag ;; T/NIL
    :multi-flag ;; List of T/NIL
    :string ;; SIMPLE-STRING
    :multi-string ;; List of SIMPLE-STRING
    :file ;; PATHNAME
    :existing-file ;; PATHNAME that must exist
    :multi-file ;; List of PATHNAME
    :multi-existing-file ;; List of PATHNAME, all of which must exist
    :integer ;; INTEGER
    :float)) ;; DOUBLE-FLOAT

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Condition Definitions
;;;

(define-condition argument-error (simple-error)
  ()
  (:documentation "A condition that is raised during argument parsing when an invalid argument or
value is encountered."))

(define-condition command-error (argument-error)
  ()
  (:documentation "Base condition class for errors related to the CMD-PARSER class."))

(define-condition no-command-error (command-error)
  ()
  (:documentation "A condition that is raised when a CMD-PARSER is used, but no command is
specified."))

(define-condition unknown-command-error (command-error)
  ()
  (:documentation "A condition that is raised when a CMD-PARSER is used, and an unknown command is
specified."))

(define-condition duplicate-command-error (command-error)
  ()
  (:documentation "A condition that is raised when two commands with the same name are added to a
CMD-PARSER."))

(define-condition argdef-error (simple-error)
  ()
  (:documentation "A condition that is raised when an argument definition is invalid."))

(defmacro argument-error (msg &rest fmt-args)
  "A convenience macro that raises an ARGUMENT-ERROR and sets the text by
FORMATting MSG."
  `(error 'argument-error :format-control ,msg :format-arguments (list ,@fmt-args)))

(defmacro argdef-error (msg &rest fmt-args)
  "A convenience macro that raises an ARGUMENT-ERROR and sets the text by
FORMATting MSG."
  `(error 'argdef-error :format-control ,msg :format-arguments (list ,@fmt-args)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Utility Functions
;;;

(defun get-implementation-args (&optional (keep-first-arg t))
  "Returns the command line arguments passed to the Lisp implementation.

Note: when using Clisp, the string 'clisp-binary' is implicitly
appended to the front of the return value.  See the manual for details
why."
  (let ((ret
         #+:sbcl sb-ext:*posix-argv*
         #+:clisp ext:*args*
         #+ccl ccl:*command-line-argument-list*
         ))
    #+clisp (setf ret (nconc (list "clisp-binary") ret))
    (if keep-first-arg
        ret
        (rest ret))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Other early stuff
;;;

(defgeneric default-help-printer (parser)
  (:documentation "Default help printer."))

(defgeneric default-version-printer (parser)
  (:documentation "Default help printer."))

(defgeneric parse-arguments (parser argv &key no-positional-arguments &allow-other-keys)
  (:documentation "Parses ARGV using PARSER.  Note: It's assumed that the 0th element in ARGV is
the binary name, like in C.

If NO-POSITIONAL-ARGUMENTS is truthy, then no positional arguments are allowed.
NO-POSITIONAL-ARGUMENTS and ALLOW-DASH-DASH cannot both be truthy at the same
time.

Specific methods may have additional &KEY arguments.  For example, the method
that is defined for the PARSER class has these:

* If DONT-QUIT is non-NIL, then the implicit help and version arguments do not
  cause the program to exit.  If DONT-QUIT is a subtype of ERROR, then a
  condition o f that type will instead be raised.
* If DONT-CHECK-HELP-OR-VER is non-NIL, and --help or --version is used, the
  help printer/version printer functions are not called."))
