;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/argparser)

;;;;
;;;; Command Parser
;;;;

(defining-consts
  (+default-help-command+ "help" :documentation "The default name of the 'help' command in a CMD-PARSER.")
  (+default-version-command+ "version" :documentation "The default name of the 'version' command in a CMD-PARSER."))

;;;
;;; API
;;;

(defgeneric cmd-parser-get (parser name)
  (:documentation "Gets the FUNCTION associated with the command NAME from PARSER, or NIL if no
command exists with that name."))

(defgeneric cmd-parser-get-data (parser name)
  (:documentation "Gets any data that will be passed to FUNCTION when the command NAME is used.
Returns two values:

1. The actual data.
2. T if the command was defined with data, or NIL otherwise."))

(defgeneric cmd-parser-add (parser name function &key help data)
  (:documentation "Associates the command NAME with FUNCTION within PARSER.  This function will be
called when the command is found during parsing.  If HELP is provided, it must
be a STRING, and will be used to display help information for the command.

FUNCTION must be a function that can handle at least one argument.  The first
argument will always be a LIST containing the unparsed command line that
appeared after the command name.  When DATA is provided, then FUNCTION will also
be called with a second argument: DATA.

If the command NAME already exists within the command parser, a
DUPLICATE-COMMAND-ERROR is raised.  A restart with the name REDEFINE-COMMAND
will be active in this situation."))

(defgeneric cmd-parser-add! (parser name function &key help data)
  (:documentation "This is the same as CMD-PARSER-ADD, except that it will always overwrite
existing command definitions."))

(defgeneric cmd-parser-delete (parser name)
  (:documentation "Deletes any command named NAME from PARSER.  Returns T if NAME existed in
PARSER, or NIL otherwise."))

(defgeneric cmd-parser-get-help (parser name)
  (:documentation "Returns the help string, if any, associated with the command NAME in PARSER."))

(defgeneric cmd-parser-commands (parser)
  (:documentation "Returns a LIST containing all of the command names in PARSER."))

(defgeneric parse-command (parser argv)
  (:documentation "Parses commands in ARGV using PARSER.

Note: It's assumed that the 0th element in ARGV is the binary name, like in C."))


;;;
;;; Class and Record Definitions
;;;

(defstruct (cmd-parser-command (:conc-name %cmd-))
  "Used internally to store function data for a command."
  (function nil :type (or null function))
  (use-data nil :type boolean)
  (data nil))

(defclass cmd-parser ()
  ((commands
    :initform (make-hash-table :test 'equalp :size 3)
    :type hash-table
    :documentation "The internal mapping of command names to PARSER instances.")

   (command-helps
    :initform (make-hash-table :test 'equalp :size 3)
    :type hash-table
    :documentation "The internal mapping of command names to their help text.")

   (case-insensitive?
    :initarg :case-insensitive
    :initform t
    :type boolean
    :reader cmd-parser-case-insensitive-p
    :documentation "When T, then command names will be handled in a case-insensitive way.  Otherwise
command names are case sensitive.

When this is T, command names are always downcased internally.")

   (help-printer
    :initarg :help-printer
    :initform #'default-help-printer
    :type function
    :reader cmd-parser-help-printer
    :documentation "The function that will be called when the built-in 'help' command is
encountered.  This must accept one argument, the CMD-PARSER instance.")

   (version-printer
    :initarg :version-printer
    :initform #'default-version-printer
    :type function
    :reader cmd-parser-version-printer
    :documentation "The function that will be called when the built-in 'version' command is
encountered.  This must accept one argument, the CMD-PARSER instance.")

   (program-name
    :initarg :program-name
    :initform ""
    :type string
    :reader cmd-parser-program-name
    :documentation "The simple name of the program (usually this is the binary name for command line
programs).  This is mainly used by the default help printing function.")

   (program-fancy-name
    :initarg :program-fancy-name
    :initform ""
    :type string
    :reader cmd-parser-program-fancy-name
    :documentation "The fancy name of the program.  This is mainly used by the default version
printing function.")

   (program-version
    :initarg :program-version
    :initform "v???"
    :type string
    :reader cmd-parser-program-version
    :documentation "The version of the program.  This is mainly used by the default version printing
function.")

   (help-pre-text
    :initarg :help-pre-text
    :initform nil
    :type (or null string)
    :reader cmd-parser-help-pre-text
    :documentation "Used by the default help printer.  This is text that will be printed after the
usage line (followed by a newline), but before the list of all possible
commands, if this is non-NIL and not an empty string.  The exact output format
used by the default printer is:

<usage line>

<HELP-PRE-TEXT>
Available Commands:
<list of all commands>
<HELP-POST-TEXT>")

   (help-post-text
    :initarg :help-post-text
    :initform nil
    :type (or null string)
    :reader cmd-parser-help-post-text
    :documentation "Used by the default help printer.  This is text that will be printed after the
list of all possible commands if this is non-NIL and not an empty string.  The
exact output format used by the default printer is:

<usage line>

<HELP-PRE-TEXT>
Available Commands:
<list of all commands>
<HELP-POST-TEXT>")

   (version-post-text
    :initarg :version-post-text
    :initform nil
    :type (or null string)
    :reader cmd-parser-version-post-text
    :documentation "Used by the default version printer.  This is text that will be printed after
the version information is printed when this is non-NIL and not an empty string.
The exact output format used by the default printer is:

<program name> <program version>
<VERSION-POST-TEXT>")

   (usage-string
    :initarg :usage-string
    :initform nil
    :type (or null string)
    :reader cmd-parser-usage-string
    :documentation "When this is non-NIL, then this will be used by the default help printer as an
alternate 'Usage:' line instead of the default built-in text."))
  (:documentation "The CMD-PARSER class is used to implement a 'command' system, where a command (a
string) maps to a PARSER."))

(defmethod initialize-instance :after ((parser cmd-parser) &key &allow-other-keys)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-slots (program-name program-version help-pre-text help-post-text version-post-text usage-string
               help-printer version-printer case-insensitive?)
      parser
    (check-type program-name string)
    (check-type program-version string)
    (check-type help-pre-text (or null string))
    (check-type help-post-text (or null string))
    (check-type version-post-text (or null string))
    (check-type usage-string (or null string))
    (check-type help-printer function)
    (check-type version-printer function)

    ;; Normalize to BOOLEAN
    (setf case-insensitive? (if case-insensitive? t nil))

    ;; Add default commands
    (cmd-parser-add parser
                    +default-help-command+
                    #'(lambda (args)
                        (declare (ignore args))
                        (muffling (funcall help-printer parser))
                        (exit 0))
                    :help "Shows this help text.")

    (cmd-parser-add parser
                    +default-version-command+
                    #'(lambda (args)
                        (declare (ignore args))
                        (muffling (funcall version-printer parser))
                        (exit 0))
                    :help "Displays version information")))

;;;
;;; Methods and Functions
;;;

(defmethod default-help-printer ((parser cmd-parser))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type parser cmd-parser)

  (with-typed-slots ((hash-table commands command-helps)
                     (string program-name)
                     ((or null string) usage-string help-pre-text help-post-text))
      parser
    ;; Get a list of command names, and determine the length of the longest one.
    (let* ((names (loop for name being the hash-keys in commands
                        collect name into ret
                        finally (when (zerop (length ret))
                                  (argdef-error "No commands in CMD-PARSER"))
                                (return ret)))
           (longest-long (loop for name string in names maximizing (length name) fixnum)))
      (declare (type list names)
               (type (and unsigned-byte fixnum) longest-long))

      (with-output-to-string (out)
        ;; Write usage line
        (if usage-string
            (format out "~a~%~%" usage-string)
            (format out "Usage: ~a <command> [options]~%~%" program-name))

        ;; Write any help pre-text
        (when (and help-pre-text (not (empty-string-p! help-pre-text)))
          (format out "~a~%" help-pre-text))

        ;; Print the commands and their help strings
        (format out "Available Commands:~%")
        (dolist (name names)
          (declare (type string name))

          (format out "  * ~a" name)
          (dotimes (i (- longest-long (length name)))
            (write-char #\Space out))
          (write-string " : " out)
          (when-var (help (gethash name command-helps))
            (unless (empty-string-p! help)
              (print-indented-string help (+ longest-long 4 3) :stream out)))
          (write-char #\Newline out))

        ;; Write any help post-text
        (when (and help-post-text (not (empty-string-p! help-post-text)))
          (format out "~a~%" help-post-text))

        ;; Send to stdout
        (format *standard-output* "~a" (get-output-stream-string out)))))
  nil)

(defmethod default-version-printer ((parser cmd-parser))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (with-typed-slots ((string program-name program-fancy-name program-version)
                     ((or null string) version-post-text))
      parser
    (format *standard-output* "~a ~a~%"
            (if (empty-string-p! program-fancy-name)
                program-name
                program-fancy-name)
            program-version)

    (when (and version-post-text (not (empty-string-p version-post-text)))
      (format *standard-output* "~a~%" version-post-text)))
  nil)

(defmethod cmd-parser-get ((parser cmd-parser) (name string))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((hash-table commands)
                     (boolean case-insensitive?))
      parser
    (multiple-value-bind (ret found?)
        (gethash (if case-insensitive?
                     (string-downcase name)
                     name)
                 commands)
      (when found?
        (%cmd-function ret)))))

(defmethod cmd-parser-get-data ((parser cmd-parser) (name string))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((hash-table commands)
                     (boolean case-insensitive?))
      parser
    (multiple-value-bind (ret found?)
        (gethash (if case-insensitive?
                     (string-downcase name)
                     name)
                 commands)
      (when found?
        (values (%cmd-data ret)
                (%cmd-use-data ret))))))

(defmethod cmd-parser-add ((parser cmd-parser) (name string) (function function) &key help (data nil data-supplied-p))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type help (or null string))

  (with-typed-slots ((hash-table commands command-helps)
                     (boolean case-insensitive?))
      parser
    (let ((name-str (if case-insensitive?
                        (string-downcase name)
                        name)))
      ;; Check for duplicate commands
      (when (hash-table-contains-p commands name-str)
        (restart-case
            (error 'duplicate-command-error :format-control "Duplicate command name: ~s"
                                            :format-arguments (list name-str))
          (redefine-command ()
            :report "Redefine the command")))

      ;; Add the command
      (setf (gethash name-str commands) (make-cmd-parser-command :function function
                                                                 :data data
                                                                 :use-data data-supplied-p))

      ;; Add the help string if we have any
      (when help
        (setf (gethash name-str command-helps) help))))
  t)

(defmethod cmd-parser-add! (parser name function &key help (data nil data-supplied-p))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (handler-bind
      ((duplicate-command-error
         (lambda (err)
           (when-var (restart (find-restart 'redefine-command err))
             ;; Redefine the command
             (invoke-restart restart)))))
    (if data-supplied-p
        (cmd-parser-add parser name function :help help :data data)
        (cmd-parser-add parser name function :help help))))

(defmethod cmd-parser-delete ((parser cmd-parser) (name string))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((hash-table commands command-helps)
                     (boolean case-insensitive?))
      parser
    (let ((name-str (if case-insensitive?
                        (string-downcase name)
                        name)))
      (cond
        ((hash-table-contains-p commands name-str)
         ;; Remove the command from both the COMMANDS table and the
         ;; COMMAND-HELPS table.
         (remhash name-str commands)
         (remhash name-str command-helps)
         t)
        (t
         nil)))))

(defmethod cmd-parser-get-help ((parser cmd-parser) (name string))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((hash-table command-helps)
                     (boolean case-insensitive?))
      parser
    (multiple-value-bind (ret found?)
        (gethash (if case-insensitive?
                     (string-downcase name)
                     name)
                 command-helps)
      (when found?
        ret))))

(defmethod cmd-parser-commands ((parser cmd-parser))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((hash-table commands)) parser
    (loop for name string being the hash-keys in commands
          collect name)))

(defmethod parse-command ((parser cmd-parser) (argv list))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  ;; Check to see if we even have a command.
  (when (<= (length argv) 1)
    (error 'no-command-error :format-control "No command given"))

  (with-typed-slots ((hash-table commands)
                     (boolean case-insensitive?))
      parser
    ;; Get the command name.
    (let ((cmd-name (if case-insensitive?
                        (string-downcase (the string (cadr argv)))
                        (cadr argv))))
      (declare (type string cmd-name))

      (multiple-value-bind (cmd found?)
          (gethash cmd-name commands)
        (if found?
            (muffling
              ;; Call the command's function
              (if (%cmd-use-data cmd)
                  (funcall (%cmd-function cmd) (cddr argv) (%cmd-use-data cmd))
                  (funcall (%cmd-function cmd) (cddr argv))))

            ;; Uh oh, not a known command
            (error 'unknown-command-error :format-control "Unknown command: ~a"
                                          :format-arguments (list cmd-name))))))
  nil)
