;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/argparser)

;;;;
;;;; ARGUMENT Class
;;;;

(defgeneric (setf arg-value) (value arg)
  (:documentation "Sets the internal value of ARG.  Note that this does NOT do any type checking,
and does not check to see if VALUE is an acceptable value for ARG's
T/ARGUMENT-TYPE.  This is to allow you more flexibility after parsing is
finished."))

(defclass argument ()
  ((name
    :initarg :name
    :type string
    :reader arg-name
    :documentation "The long name for the argument (minus the '--').  This cannot be empty.")

   (short-name
    :initarg :short-name
    :initform nil
    :type (or null character)
    :reader arg-short-name
    :documentation "The short name for the argument (minus the '-'), or NIL if this argument does
not use a short name.")

   (help
    :initarg :help
    :initform ""
    :type string
    :reader arg-help
    :documentation "A short string describing what the argument does.")

   (group
    :initarg :group
    :initform ""
    :type string
    :reader arg-group
    :documentation "A group this argument belongs to.")

   (type
    :initarg :type
    :initform :flag
    :type t/argument-type
    :reader arg-type
    :documentation "The T/ARGUMENT-TYPE of the argument.")

   (init-value
    :initarg :init-value
    :reader arg-init-value
    :documentation "The initial value of the argument.  When non-NIL, then the type of this value
must be valid for the T/ARGUMENT-TYPE of this argument.")

   (constraint
    :initarg :constraint
    :initform nil
    :reader arg-constraint
    :documentation "When non-NIL, then this is used to constrain the acceptable set of values for
this argument.  The exact behavior of this slot depends on the TYPE of the
argument:

* :FLAG - CONSTRAINT is always ignored.

* :MULTI-FLAG - CONSTRAINT is always ignored.

* :STRING - CONSTRAINT should be a list, and the argument's value after parsing
  must be one of the given values.  Always case sensitive.

* :MULTI-STRING - CONSTRAINT should be a list, and each argument value after parsing
  must be one of the given values.  Always case sensitive.

* :FILE - CONSTRAINT is always ignored.

* :EXISTING-FILE - CONSTRAINT is always ignored.

* :MULTI-FILE - CONSTRAINT is always ignored.

* :MULTI-EXISTING-FILE - CONSTRAINT is always ignored.

* :INTEGER - CONSTRAINT should be a CONS, where the CAR is the minimum allowed
  value, and the CDR is the maximum allowed value, inclusive.

* :FLOAT - CONSTRAINT should be a CONS, where the CAR is the minimum allowed
  value, and the CDR is the maximum allowed value, inclusive.")

   (value
    :initform nil
    :reader arg-value
    :documentation "The value of the argument after parsing.  This is always NIL or the value of
INIT-VALUE before parsing is performed.")

   (called?
    :initform nil
    :type boolean
    :documentation "Internal flag indicating whether or not this argument was called.  If you want
to see if the argument was actually called, use ARG-CALLED-P.")

   (called-as
    :initform ""
    :type string
    :documentation "The string used to call the argument, including the hyphens.  For example, if
the argument has a short name of #\F, and the argument's short name was used on
the command line, then this will be the string \"-F\" after parsing is
completed.")

   (checking-fn
    :initarg :checking-fn
    :initform nil
    :type (or null function)
    :reader arg-checking-fn
    :documentation "When non-NIL, then this function will be called when the argument is encountered
during parsing, with the expectation that it will check the argument value for
validity.  Essentially this allows you to customize the parsing of an argument's
value.

When non-NIL, this must be a function that should take an ARGUMENT parameter.

Note: having this function raise an ARGUMENT-ERROR is the correct way to
indicate that CHECK-FN has failed.")

   (not-found-str
    :initarg :not-found-str
    :initform nil
    :type (or null string)
    :reader arg-not-found-str
    :documentation "When non-NIL, and the argument is not encountered on the command line when
parsing is performed, then an ARGUMENT-ERROR will be raised with this exact
string once parsing is completed.

Essentially, this is one way you can define a required argument."))
  (:documentation "The ARGUMENT class is used to define a single argument for a parser.  An
instance should *always* be created using DEFARGUMENT."))

(defmethod print-object ((obj argument) out)
  (print-unreadable-object (obj out :type t)
    (format out "~a~a, type: ~a, Group: ~a"
            (arg-name obj)

            (if (arg-short-name obj)
                (concatenate 'string " / -" (string (arg-short-name obj)))
                "")

            (arg-type obj)

            (if (and (arg-group obj) (not (string= (arg-group obj) "")))
                (arg-group obj)
                "none"))))

(define-typed-fn check-arg-value-against-constraint ((argument arg) value)
    (boolean t)
  "Checks to see if VALUE is valid for ARG, given ARG's constraints.  Returns T
if it is, or NIL otherwise.  If the argument has no constrants, this always
returns T.

VALUE is expected to already be in the correct type (e.g. an integer for an
:INTEGER argument)."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (if (arg-constraint arg)
      (ecase (the t/argument-type (arg-type arg))
        ;; These do not use constraints.
        ((:flag :multi-flag :file :existing-file :multi-file :multi-existing-file)
         t)
        (:integer
         (and (integerp value)
              (>= value (car (the cons (arg-constraint arg))))
              (<= value (cdr (the cons (arg-constraint arg))))))
        (:float
         (muffling
           (and (floatp value)
                (>= value (car (the cons (arg-constraint arg))))
                (<= value (cdr (the cons (arg-constraint arg)))))))
        ((:string :multi-string)
         (when (and (stringp value)
                    (muffling (find value (the list (arg-constraint arg)) :test #'string=)))
           t)))
      t))

(defmethod initialize-instance :after ((arg argument) &key &allow-other-keys)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (unless (slot-boundp arg 'name)
    (argdef-error "No long name specified"))

  (with-slots (name short-name help group type constraint checking-fn not-found-str)
      arg
    (check-type name string)
    (check-type short-name (or null character))
    (check-type help string)
    (check-type group string)
    (check-type type t/argument-type)
    (check-type checking-fn (or null function))
    (check-type not-found-str (or null string))

    ;; Check argument names
    (cond
      ((empty-string-p name)
       (argdef-error "An argument must have a long name"))

      ((string-starts-with name "-")
       (argdef-error "An argument's long name cannot start with a hyphen"))

      ((characterp short-name)
       (cond
         ((muffling (string= name (string short-name)))
          (argdef-error "An argument's short name cannot be the same as its long name"))

         ((digit-char-p short-name)
          (argdef-error "An argument cannot have its short name be a number"))

         ((char= short-name #\-)
          (argdef-error "An argument's short name cannot be a hyphen"))

         ((not (alpha-char-p short-name))
          (argdef-error "Invalid argument short name: ~w" short-name)))))


    ;; Check constraint based on the type
    (ecase type
      ((:flag :multi-flag :file :multi-file :existing-file :multi-existing-file)
       (unless (null constraint)
         (warn "CONSTRAINT is ignored for arguments of type ~a" type)))

      ((:string :multi-string)
       (unless (null constraint)
         (unless (listp constraint)
           (argdef-error "Arguments of type ~a, such as ~a, can only have a constraint of type LIST" type name))
         (unless (every #'stringp constraint)
           (argdef-error "Arguments of type ~a, such as ~a,  must have constraints where every value is a STRING"
                         type name))))

      (:integer
       (unless (null constraint)
         (unless (consp constraint)
           (argdef-error "Arguments of type ~a, such as ~a,  can only have a constraint of type CONS"
                         type name))
         (unless (and (integerp (car constraint))
                      (integerp (cdr constraint)))
           (argdef-error "Arguments of type ~a, such as ~a,  must have constraints where both the CAR and ~
CDR are integers"
                         type name))))

      (:float
       (unless (null constraint)
         (unless (consp constraint)
           (argdef-error "Arguments of type :FLOAT, such as ~a,  can only have a constraint of type CONS" name))
         (unless (and (typep (car constraint) '(or float rational integer))
                      (typep (cdr constraint) '(or float rational integer)))
           (argdef-error "Arguments of type :FLOAT, such as ~a,  must have constraints where both the ~
CAR and CDR are floats, rationals, or integers"
                         name)))))

    ;; Check initial value if it's present
    (when (slot-boundp arg 'init-value)
      (with-slots (init-value)
          arg
        (case type
          (:multi-string
           (when init-value
             (if (and (typep init-value 'list)
                      (every #'stringp init-value))
                 ;; It's a LIST and has only STRINGs.  Check the string values.
                 (dolist (str init-value)
                   (unless (check-arg-value-against-constraint arg str)
                     (argdef-error "The initial value ~s is not valid according to the constraints." init-value)))

                 ;; Not the right type.
                 (argdef-error "The initial value for a :MULTI-STRING argument must be a list of strings."))))

          (otherwise
           (unless (check-arg-value-against-constraint arg init-value)
             (argdef-error "The initial value ~s is not valid according to the constraints."))))

        ;; Now store the init-value in the value slot
        (setf (slot-value arg 'value) init-value)))))

(defmethod (setf arg-value) (value (arg argument))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (setf (slot-value arg 'value) value))
