(require 'asdf)
(asdf:load-system :cl-sdm-semver)
(in-package :asdf-user)

(defun apply-block-compilation (thunk)
  "Applies block compilation if using SBCL, and NEVER-APPLY-BLOCK-COMPILATION is
not in *FEATURES*."
  #-never-apply-block-compilation
  (apply thunk (when (uiop:featurep :sbcl) '(:block-compile t)))

  #+never-apply-block-compilation
  (apply thunk (when (uiop:featurep :sbcl) '(:block-compile nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; SDM-SYSTEM
;;;
;;; The SDM-SYSTEM class is a subclass of ASDF:SYSTEM that expands its features.
;;;

(defclass sdm-system (asdf:system)
  ((semantic-version
    :initarg :semantic-version
    :initform nil
    :accessor system-semantic-version
    :documentation "The Semantic Version of this system.  This is separate from
the :VERSION key.  When doing version checking, if a full Semantic Version
comparison can't be done, then version checking will fall back to using the
:VERSION key for this system.")))

(defmethod initialize-instance :after ((sys sdm-system) &key &allow-other-keys)
  (with-slots (semantic-version)
      sys
    (etypecase semantic-version
      (string (setf semantic-version (sdm-semver:parse-semantic-version semantic-version)))
      (sdm-semver:semver t)
      (null t))))

(defmethod asdf:version-satisfies :around ((sys sdm-system) version)
  (with-slots (semantic-version)
      sys
    (if (and semantic-version
             (or (stringp version)
                 (typep version 'sdm-semver:semver)))
        ;; We can try to do a full Semantic Version comparison on the two
        ;; versions.
        (handler-bind
            ((sdm-semver:semver-error
               ;; Fallback to old style version checking if an error occurs.
               (lambda (err)
                 (declare (ignore err))
                 (warn "Could not check Semantic Version of system ~a against ~s" sys version)
                 (when (next-method-p)
                   (call-next-method)))))

          ;; Get our two SEMVER instances.
          (let ((ver1 (typecase semantic-version
                        (sdm-semver:semver semantic-version)
                        (string (sdm-semver:parse-semantic-version semantic-version)) ;; Just in case
                        (otherwise
                         (when (next-method-p)
                           ;; Fallback to old style version checking.
                           (return-from asdf:version-satisfies (call-next-method))))))

                (ver2 (typecase version
                        (string (sdm-semver:parse-semantic-version version))
                        (sdm-semver:semver version)
                        (otherwise
                         (when (next-method-p)
                           ;; Fallback to old style version checking.
                           (return-from asdf:version-satisfies (call-next-method)))))))
            (sdm-semver:semver<= ver2 ver1)))

        ;; We can't do a full Semantic Version comparison.  Fallback.
        (when (next-method-p)
          (call-next-method)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(asdf:defsystem cl-sdm-asdf
  :class :sdm-system
  :long-name "CL-SDM (ASDF Extras)"
  :description "Extra batteries for ASDF"
  :version #.(nth 1 (uiop:read-file-lines (uiop:subpathname *load-pathname* "version.txt")))
  :semantic-version #.(nth 0 (uiop:read-file-lines (uiop:subpathname *load-pathname* "version.txt")))
  :licence "GNU Affero General Public License Version 3 or Later"

  :maintainer "Remilia Scarlet <remilia@posteo.jp>"
  :author "Remilia Scarlet <remilia@posteo.jp>"
  :homepage "https://chiselapp.com/user/MistressRemilia/repository/cl-sdm/"
  :source-control (:fossil "https://chiselapp.com/user/MistressRemilia/repository/cl-sdm/")

  :defsystem-depends-on (#:cl-sdm-semver))
