#ifndef index_storage_H
#define index_storage_H
/*
 * Copyright 2022, ttcoder
 * All rights reserved. Distributed under the terms of the MIT license.
 */


// base.a
#include <base/env.h>
#include <base/heap.h>  // member

// project
#include "SlotIO.h"  // member



class IndexStore
{
public:

		IndexStore( Genode::Env & env );
		
		// -- Snooping (to index) --
		
		// ops for files/dirs (inside an FS)
		void NodeDeleted( const char * fullpath );
		void NodeMoved( const char * from, const char * to );
		
		// ops for attributes (inside a file/dir)
		void AttrDeleted( const char * raw_attrname, const char * from_node );
		void AttrMutated(
			const char * path,
			const char * raw_attrname,
			const void * buf,
			long buf_size
			);
		
		
		// -- Load/save index --
		
		void SetAs_And_Check( const void * index_contents, unsigned size );
		
		const char * IndexBuffer() const;
		unsigned IndexBufferSize() const;
		
		
		// -- Querying (from index) --
		
		void SetQuery( const char * predicate );
		
		const void * ResultsBuf() const;
		int64        ResultsSize() const;
			///xx As a VFS plug-in, are we re-entrant ? or should we guard against concurrent query accesses, by combining SetQuery() and Results*() into just one atomic call ?
			///ToDo: as an immediate measure, I could add a bool flag and flip it on/off when entering/leaving, and assert() that we don't enter with the flag already "up" ; if the assert() triggers, I'll have the answer to my question


private:  // Data

		// support for below
		Genode::Heap memHeap;
		
		// index data
		SlotIO slotIO;
		
		// query results
		BufIO resultsIO;
		
		/*
			-- Notes on the SlotIO-produced index file --
			
			It's a text file that looks like this:
			
				/Music/TheWall.mp3
				<tab> Audio:Artist=RTSCPink Floyd
				/Music/ComfortablyNumb.mp3
				<tab> Audio:Artist=RTSCRoger Waters
				<tab> Audio:Album=RTSCThe Wall
			
			Keeping it textual facilitates inspection and debugging, and there is
			no benefit to making it binary. Indeed the file paths are expected to
			be textual (no embedded \0 etc hopefully), and so are the attribute
			names and values -- in fact we support the same set of attribute types
			as BeOS & Haiku, they have that same policy, to wit:
			
				$ uname -a
				Haiku shredder 1 hrev55181+51 Jul 22 2021 12:18: x86_64 x86_64 Haiku
				$ /bin/mkindex
				Usage: mkindex [options] <attribute>
				(..)
				-t, --type=TYPE       the type of the attribute being indexed.  One of "int", "llong", "string", "float", or "double".
			
			It makes sense, nobody will ever "query" for binary data. Thus, this
			allows me to keep implementation super-simple, and to give up on my (scary)
			initial plans of having the index grow by 1024 bytes "blocks" allocated
			piecemeal with the index size being a multiple of <blocksize> etc.
		*/
};


#endif // index_storage_H

